///////////////////////////////////////////////////////////////////////////////	
//	Module			: pif2.c 
//	Description		: Linux device driver for PIF-II (LP-042) 
//
//	Compiler		: gcc 4.4.7++ 
//	Author			: T.Matsuzawa of JEI Matsumoto Lab.
//	E-mail			: tmatsu.jei@nifty.com
//- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
// Maintenance:
//	<Date>		 <Version>	<Desc.>
//	Mar.12,'14	  1.2.0		Renewal for CentOS6.5/64bit
//	Apr. 8,'14	  1.2.0		Unsupported DMA
//	May 28,'14	  1.3.0		Supported Fedora20
//	Sep.18,;15	  1.3.1		Fixed a bug in uart_read & uart_write
///////////////////////////////////////////////////////////////////////////////	
#include	<linux/kernel.h>
#include	<linux/module.h>
#include	<linux/vmalloc.h>
#include	<linux/delay.h>
#include	<linux/pci.h>
#include	<linux/mm.h>
#include	<linux/sched.h>
#include	<linux/wait.h>
#include	<linux/interrupt.h>
#include	<linux/cdev.h>
#include	<linux/fs.h>
#include	<linux/version.h>
#include    <linux/uaccess.h>
#include	<asm/io.h>
//#include	<asm/uaccess.h>

//#include	"../../include/pifg/common.h"
#include	"common.h"
#include	"../../include/drivers/pif2.h"
#include	"../../include/drivers/pif_api.h"
#include	"pci9054.h"			// definition of PCI9054  
#include	"xr16L2750.h"		// definition of EXAR XR16L2750  
//#include <linux/wait.h>
//#include <wait.h>
//#include <sys/wait.h>
//#include <linux/wait.h>
//#include <linux/wait.h>
//#include <linux/wait.h>
 #include    <linux/pagemap.h>       // added on Dec.13, 2017 by JEI for F27_64bit

MODULE_LICENSE("GPL");
#ifndef	true
#define	true	1
#define	false	0
#endif

#define	FEDORA20	1

//////////////////////////////////////////////////////////////////////////////
//	Prototypes
//////////////////////////////////////////////////////////////////////////////
//	Global Functions ________________________________________________________

int 	init_module(void);				// install module
void 	cleanup_module(void);			// remove module

//	External Functions ______________________________________________________

void	initUART(unsigned* uartaddr);	// Initialize UART
ssize_t uart_read(struct file *file,char *buf,size_t count,loff_t *off);
ssize_t uart_write(struct file *file,const char *buf,size_t count,loff_t *off);
void 	isrUART(int port);				// ISR for UART 
int		getRxCnt(int port);				// get RX data count
int		getPAGEND(void);					// get PAGEND count
void	releaseWait(void);				// release waits
void	resetBuffer(int port);			// reset UART buffer
void	setBaud(int baud);				// set baud rate of channel-B
void	setTrigger(int trigger);		// set trigger level of channel-B
void	selectChannel(int ch);			// select channel

//	Local Functions _________________________________________________________

//		System calls
static int		pif_open(struct inode *inode, struct file *file);
static int		pif_close(struct inode *inode, struct file *file);
#if	FEDORA20
static long 	pif_ioctl(struct file *file, uint cmd, ulong value);
#else
static int 		pif_ioctl(struct inode *inode,struct file *file, uint cmd, ulong value);
#endif
static int 		pif_mmap(struct file *file, struct vm_area_struct *vma);

//		Misc.
static int 	findPif(void);					// find PIF-II
static void resetPif(void);					// reset whole PIF-II
static void resetUart(void);				// reset UART
static void resetVideo(void);				// reset video circuit
static int	waitVideoRequest(void);			// wait for video data request 
static int	waitPWAR(void);
static int	waitPERR(void);
static int	readRegister(REG* reg);
static void	writeRegister(REG* reg);
//static int inspect(ulong ndx);

//		Interrupt handlers
// Kernel-2.6 FC5/CentOS 5
irqreturn_t pif_interrupt(int irq, void *);


//	Global variables ________________________________________________________
//
//	File Operations
//
struct file_operations pif_fops = {
	.owner =	THIS_MODULE,
	.read =		uart_read,				// read
	.write =	uart_write,				// write
#if	FEDORA20
	.unlocked_ioctl =	pif_ioctl,		// ioctl
#else
	.ioctl =	pif_ioctl,				// ioctl
#endif
	.open =		pif_open,				// open
	.release =	pif_close,				// close
	.mmap =		pif_mmap,				// mmap
};

int	iPAGEND;                			// PAGEND counter
int	iPGEDelivered;          			// PAGEND counter deliverd to user
int	iIntCount[13];						// interrupt counters

//	Local variables __________________________________________________________

static int		PIFTYPE = 2;			// PIF-II
static int		VERSION	= 0x0200;		// PIF-II
static char		version[] = "Version 2.0.0 (2017-12-13)";
static int		devid = PIF_DEVICE_ID;	// device id
static uchar	cIrq;					// IRQ Number

/*	Address Space
	This module uses the following three address space
	Base address 0:	base[0]:*pPCIREG	PLX PCI9054 Locar Registers
	Base address 2:	base[1]:*pPIFREG	PIF Registers
	Base address 3:	base[2]:			Video data output FIFO 
*/
static unsigned long 	base[3];		// Base Address (physical address) 
static unsigned	*pPCIREG;				// PCI9054 Register Address remapped
static unsigned	*pPIFREG;				// PIF Register Address remapped
static unsigned	*pBANK;					// Page Bank area remmaped

static int		iIntPERR;				// PEI counter
static int		iIntPWAR;				// PWI counter
static int		iPWARUser;				// PWR counter (user)
static int		iIntPWR;				// PWR counter (kernel)
static int		iPWRUser;				// PWR counter (user)

static  DECLARE_WAIT_QUEUE_HEAD(pwrq);	//  wait queue for PWR
static  DECLARE_WAIT_QUEUE_HEAD(warq);	//  wait queue for WAR

static dev_t dev_id = 0;
static struct cdev c_dev; // character device structure

static unsigned major = 30;
static unsigned minor = 0;
static unsigned num = 1;

#define MINOR_COUNT	1
#define MODNAME "pifII"

//	Global Functions _________________________________________________________
/*
	name:	init_module	
	desc:	install device driver	
	note:	This routine is called from insmod. So don't rename this routine.
*/
int init_module()
{
	int	ret;

	if(!(ret = findPif())){					// find a PIF on the pci slot 
        ret = alloc_chrdev_region(&dev_id, minor, MINOR_COUNT, MODNAME);
        if(ret < 0){
            printk("alloc_chrdev_region failed\n");
            return ret;
        }
        major = MAJOR(dev_id);
        dev_id = MKDEV(major,minor);
        ret = register_chrdev_region(dev_id,num,MODNAME);
		printk("%s is loaded\n",MODNAME);
        printk("major = %d\n", major);
        printk("minor = %d\n", minor);
        cdev_init(&c_dev, &pif_fops);
        c_dev.owner = THIS_MODULE;
        c_dev.ops = &pif_fops;
        ret = cdev_add(&c_dev, dev_id, num);
        if( ret < 0 ){
            printk("cdev_add failed\n");
            return ret;
        }
    }
    return ret;
}

/*
	name:	cleanup_module	
	desc:	remove device driver	
	note:	This routine is called from rmmod. So don't rename this routine.
*/
void cleanup_module()
{
	cdev_del(&c_dev);
    unregister_chrdev_region(dev_id,MINOR_COUNT);
	printk("Pifc Device Driver uninstalled.\n");
}

// Local Functions __________________________________________________________ 
//		System calls ________________________________________________________
/*
	name:	pif_open
	desc:	open device driver for PIF-II	
	args:	struct inode *inode = pointer to inoce
			struct file *file = pointer to file	
	rtrn:	int = 0	: OK
			    < 0	: error
*/
static int 
pif_open(struct inode *inode, struct file *file)
{
	int	rtn;

//	Install interrupt handler 
	rtn = request_irq( 		//	1) Try unsharable first
		cIrq,						// IRQ
		pif_interrupt,				// interrupt handler
#if	FEDORA20
		IRQF_SHARED,				// flags 
#else
		IRQ_DISABLED,				// flags 
#endif
		"pif",						// device name
		&devid						// device id
	);
	if(!rtn){						// OK !!
    	printk("Pifc 2 Open_TDU (IRQ%d) %s\n",cIrq,version);

//	Remap PCI9054 and PIF registers(physical addr) to virtual kernel addr.   	

//		pPCIREG = ioremap(base[0], 1024);
//		pPIFREG = ioremap(base[1], 1024);
//		pBANK = ioremap(base[2], 1024);
        pPCIREG = ioremap(base[0], 1024);
		pPIFREG = ioremap(base[1], 1024);
		pBANK = ioremap(base[2], 1024);
        
//        printk("Matthew %lx \n", (unsigned long)base[0]);
//		printk("pPCIREG = %lx \n", (unsigned long)pPCIREG);
//		printk("pPIFREG = %lx \n", (unsigned long)pPIFREG);
//		printk("pBANK = %lx \n", (unsigned long)pBANK);
		resetPif();					// reset PIF 
		resetUart();				// reset UART 
		initUART(pPIFREG+16);		// initialize Uart
	}
	else
		printk("Failed to open Pifc 2(%d)\n",rtn);
	return rtn;
}

/*
	name:	pif_close
	desc:	close device driver for PIF-II	
	args:	struct inode *inode = pointer to inoce
			struct file *file = pointer to file	
	rtrn:	int = 0	: OK
			    < 0	: error
*/
static int 
pif_close(struct inode *inode, struct file *file)
{
	putPifReg(PIF_CTRL, CNTL_DEFAULT);	// mask all PIF interrupts 
	resetUart();						// reset UART 
	free_irq(cIrq, &devid);				// free IRQ
	iounmap(pPCIREG);					// free remapped area
	iounmap(pPIFREG);
	iounmap(pBANK);
    printk("Pifc Close ......\n");
	return 0;
}

/*
	name:	pif_ioctl
	desc:	ioctl commands handler		
	args:	struct inode *inode = pointer to inoce
			struct file *file = pointer to file	
			uint cmd = ioctl command
			ulong value = value to be written (in case of IOW)
	rtrn:	int	= return value (in case of IOR)	
	note:	All ioctl commands are defined in "pifc_api.h"
*/
#if	FEDORA20
static long 
pif_ioctl(struct file *file, uint cmd, ulong value)
#else
static int 
pif_ioctl(struct inode *inode, struct file *file, uint cmd, ulong value)
#endif
{
	int		iResult = 0;
    REG     reg;
    
    if(cmd == PIF_RD_REG || cmd == PIF_WT_REG) {
        iResult = copy_from_user(&reg,(void*)value,sizeof(REG));
//        if(iResult == 0)
//            printk("success!!!\n"); // removed in version 7.05-35x01, to reduce the entries to journal log.
    }

	switch(cmd){
		case PIF_RD_TYPE:		iResult = PIFTYPE; break;
		case PIF_RD_VER:		iResult = VERSION; break;
		case PIF_RST_PIF: 		resetPif(); break;
		case PIF_RST_UART: 		resetUart(); break;
		case PIF_RST_VIDEO: 	resetVideo(); break;
		case PIF_RD_IRQ: 		iResult = cIrq; break;
		case PIF_RD_INTC: 		iResult = iIntCount[value]; break;
		case PIF_RD_PERR: 		iResult = waitPERR(); break;
		case PIF_RST_PERR: 		iIntPERR = 0; break;
		case PIF_RD_PWAR: 		iResult = waitPWAR(); break;
		case PIF_RST_PWAR: 		iIntPWAR = 0; break;
		case PIF_RD_VDRQ: 		iResult = waitVideoRequest();break;
		case PIF_RD_OPRXC: 		iResult = getRxCnt(1);break;
		case PIF_WT_TRIGGER:	setTrigger(value);break;
		case PIF_WT_BAUD: 		setBaud(value);break;
		case PIF_WT_CHANNEL:	selectChannel(value);break;
		case PIF_RD_EIFRXC:		iResult = getRxCnt(0);break;
		case PIF_RD_PAGEND:		iResult = getPAGEND();break;
		case PIF_RST_PAGEND:	iPAGEND = iPGEDelivered = 0; break;
//		case PIF_RD_REG:		iResult = readRegister((REG*)value); break;
//		case PIF_WT_REG:		writeRegister((REG*)value); break;
		case PIF_RD_REG:		iResult = readRegister(&reg); break;
		case PIF_WT_REG:		writeRegister(&reg); break;
	}
	return iResult;
}

/*
	name:	pif_mmap
	desc:		
*/
static int 
pif_mmap(struct file *file, struct vm_area_struct *vma)
{
    unsigned long   dest;

    dest = base[2];
//	printk("mmap: %lx %lx\n",dest, (vma->vm_end - vma->vm_start));	// removed in version 7.05-35x01, to reduce the entries to journal log.
	if(remap_pfn_range(vma, vma->vm_start,(dest >> PAGE_SHIFT),
						vma->vm_end - vma->vm_start,vma->vm_page_prot))
		return -EAGAIN;
	return OK;
}

//
//	Interrupt Handler
//
irqreturn_t 
pif_interrupt(int irq, void *devid)
{
	unsigned	iflag;

	if(cIrq != irq) return IRQ_NONE;		// illegal IRQ
	if(!(iflag = getPifReg(PIF_ISR) & 0x3f))// get causes of interrupt 
		return IRQ_NONE;
	putPifReg(PIF_ISR, iflag);				// clear interrupt 
	iIntCount[0]++;							// increase counter				

	if(iflag & FUIA)isrUART(0);				// UART-A interrupt
	if(iflag & FUIB)isrUART(1);				// UART-B interrupt 
	if(!(getPifReg(PIF_ISR) & EACT)){		// EIF inactive?
		putPifReg(PIF_CTRL,CNTL_DEFAULT);
		//printk("Printer engine off!!\n");
		return IRQ_NONE;
	}
	if(iflag & FWAR){						// printer status changed
		iIntCount[2]++;						// increase Intr. counter
		iIntPWAR = 1;						// set flag
	}
	if(iflag & FERR){		 				// online/offline change 
		iIntCount[1]++;						// increase Intr. counter
		iIntPERR = 1;						// set flag
		wake_up(&warq);						// wake up
	}
	if(iflag & FPWR){
		iIntCount[3]++;                     // increase Intr. counter
		iIntPWR++;                          // set flag
		//printk("PWR(%d)(%d)\n", iIntPWR,iPWRUser);
		//wake_up(&pwrq);						// wake up user process
        printk("PWR(%d)(%d)\n", iIntPWR,iPWRUser);
	}
	return IRQ_HANDLED;	
}

/*
	name:	findPif	
	desc:	Find the PIF-II by using PCI BIOS	
	args:	None
	rtrn:	int = 0	: OK (Found)
			    < 0	: error
*/
static int findPif()
{
	struct	pci_dev *dev = NULL;
	int	ret;
	
//	find pifc device
#if	FEDORA20
	dev = pci_get_device(PIF_VENDOR_ID, PIF_DEVICE_ID, dev);
#else
	dev = pci_find_device(PIF_VENDOR_ID, PIF_DEVICE_ID, dev);
#endif
	if(!dev){
		printk("PIF-II not found !\n");
		return -ENODEV;
	}
	ret = pci_enable_device(dev);
	if(ret){
		printk("PIF-II not available %d !\n",ret);
		return ret;
	}

//	get IRQ (Interrupt Line)
	cIrq = dev->irq;

//	read Base addresses 
	base[0] = pci_resource_start(dev, 0);	// get Base address 0
	base[1] = pci_resource_start(dev, 2);	// get Base address 2
	base[2] = pci_resource_start(dev, 3);	// get Base address 3

	printk("Base Addr0 = %lx \n", base[0]);
	printk("Base Addr2 = %lx \n", base[1]);
	printk("Base Addr3 = %lx \n", base[2]);

	return 0;
}

/*
	name:	resetPif	
	desc: 	reset whole PIF 	
	args:	None
	rtrn:	None
*/
static void resetPif()
{
	int			i;
	unsigned	val;

//	PCI9054 software reset
	
	val = readl(CNTRL);					// read CNTRL in the runtime registers
	writel(val | RESET, CNTRL);			// set bit30 on
	udelay(100);						// wait 100 us
	writel(val, CNTRL);					// set bit30 off
	writel(val | RELOAD, CNTRL);		// set bit29 on
	udelay(100);						// wait 100 us
	writel(val, CNTRL);					// set bit29 off
	mdelay(1);							// wait 1 ms

	writel(readl(INTCSR) | LIIE, INTCSR);

//	initialize local variables

	for(i = 0; i < 13; i++)				// reset interrupt counters
		iIntCount[i] = 0;
	iIntPERR = iIntPWAR = iPWARUser = 0;
	iIntPWR = iPWRUser = 0;
    iPGEDelivered = iPAGEND = 0;
	putPifReg(PIF_CTRL, CNTL_DEFAULT);	// mask all PIF interrupts 
	val = getPifReg(PIF_ISR);
}

/*
	name:	resetUart	
	desc: 	reset UART chip 
	args:	None
	rtrn:	None
*/
static void resetUart()
{
	putPifReg(PIF_CTRL, getPifReg(PIF_CTRL) | URST);	// set URST 
	putPifReg(PIF_CTRL, getPifReg(PIF_CTRL) & ~URST);	// reset URST 
	initUART(pPIFREG+16);								// initialize Uart
}

/*
	name:	resetVideo	
	desc: 	reset video circuit along with buffers
	args:	None
	rtrn:	None
*/
static void resetVideo()
{
	putPifReg(PIF_CTRL, getPifReg(PIF_CTRL) | IRST);	// set IRST 
	udelay(100);										// wait 100us
	putPifReg(PIF_CTRL, getPifReg(PIF_CTRL) & ~IRST);	// reset IRST 
	iIntPWR = iPWRUser = 0;
}

static int waitVideoRequest()
{
	int	rtn;

	if(iIntPWR == iPWRUser){
		printk("S");
		rtn = wait_event_timeout(pwrq,(iIntPWR != iPWRUser),10 * HZ);
		if(rtn < 0)
			return -ERESTARTSYS;
		else if(!rtn){	// timeout !!
			printk("T");
			return 0;
		}
	}
	iPWRUser++;
	printk("waitVideoRequest(%d)(%d)\n", iIntPWR,iPWRUser);
	return 1;
}

static int	waitPWAR()
{
 //   wait_queue_t    wait;
    wait_queue_entry_t    wait;                     // changed on Dec.13,2017
	if(!iIntPWAR){
		init_waitqueue_entry(&wait,current);
		add_wait_queue(&warq,&wait);
		set_current_state(TASK_INTERRUPTIBLE);
		schedule_timeout(HZ);                       // wait 1 sec.
		remove_wait_queue(&warq,&wait);
		set_current_state(TASK_RUNNING);
	}
	if(iIntPWAR){
		iIntPWAR = 0;
		return true;
	}
	else
		return false;
}

static int	waitPERR()
{
 //   wait_queue_t    wait;
    wait_queue_entry_t    wait;                     // changed on Dec.13,2017
	if(!iIntPERR){
		init_waitqueue_entry(&wait,current);
		add_wait_queue(&warq,&wait);
		set_current_state(TASK_INTERRUPTIBLE);
		schedule_timeout(HZ);                       // wait 1 sec.
		remove_wait_queue(&warq,&wait);
		set_current_state(TASK_RUNNING);
	}
	if(iIntPERR){
		iIntPERR = 0;
		return true;
	}
	else
		return false;
}

static int readRegister(REG* reg)
{
	int	offset;
	int	val;

	offset = reg->offset;
	if(offset < PCIREG_MAX) 			// read PCI9054 Registers
		val = getPciReg(offset);
	else { 								// read PIF Registers
		offset -= PCIREG_MAX;
		val = getPifReg(offset);
		if(offset >= (R_UART_RHRA - PCIREG_MAX))
			val &= 0xff;				// use LSB(8)
	}
	return val;
}

static void writeRegister(REG* reg)
{
	int	offset;
	int	val;

	offset = reg->offset;
	val = reg->val;
	if(offset < PCIREG_MAX) 			// write to PCI9054 Registers
		putPciReg(offset,val);
	else{ 								// write to PIF Registers
		offset -= PCIREG_MAX;
		putPifReg(offset,val);
	}
}

/*static int inspect(ulong ndx)
{
	int n;
	
	n=(int)ndx;
	if(n == 1)
	{
		
	}
	
	return 1;
}*/
