///////////////////////////////////////////////////////////////////////////////
//	Module		: CLoader.C		
//	Description	: Loader for Motorola S-Format
//
//	Author		: T.Matsuzawa of JEI Matsumoto Lab.
//	E-mail		: matsuzawa.jei@nifty.com
//-----------------------------------------------------------------------------
//Maintenance:
//	Oct. 17, '18	1st release 
///////////////////////////////////////////////////////////////////////////////
#include	<iostream>
#include 	<fstream>
using namespace std;

#include 	<cstring>
#include	"mcUpdator.h"
#include	"CLoader.h"

#define		DEBUG	0

//	Global Methods ___________________________________________________________
/*
	Method 		: CLoader()
	Description	: Constructor
	Procedure	: 1. Allocate memory for loading 
				  2. Fill allocated memory with 0xFF (initialize)
	Arguments	: int size = memory size in bytes 
				  int offset = offset 
*/
CLoader::CLoader(int size, int offset)
{
//	1. allocate memory 
	pLM = pGet = new char [size];				// allocate memory
	nOffset = offset;							// save offset

//	2. fill allocate memory with 0xFF 
	if(pLM != NULL){
		cout << "  allocate memory " << size/1024 << " KB" << endl;
		memset(pLM,0xff,size);
	}
}

/*
	Method 		: ~CLoader()
	Description	: Destructor
	Procedure	: Free allocated memory 
*/
CLoader::~CLoader()
{
	delete pLM;
}

/*
	Method 		: loadSformat()
	Description	: Load Motrola S-Format file on the memory
	Procedure	: 1. Check if memory is allocated
				  2. Open S-Format file 
				  3. Extract S1/S2 data and convert them binary
	Arguments	: char* fname = file name
	Returns		: int =  0 : Normal completion
					  != 0 : error   
*/
int	CLoader::loadSformat(char* fname)
{
//	1. Check memory
	if(pLM == NULL){
		cout << "Could not allocate memory!\n";
		return ERR_CL_NOMEMORY;
	}

//	2. Open S-Format file
	cout << "  Open " << fname << " ....\n";
	fin.open(fname);						
	if(!fin){
		cout << "Could not open a file!\n";
		return ERR_CL_FILEOPEN;
	}

//	3. Extract S1/S2 data and convert them binary
	extractData();

#if	DEBUG
	ofstream fout("debug.tmp");
	fout.write(pLM,nSize);
	fout.close();
#endif
	return 0;
}

/*
	Method 		: getchar()
	Description	: get character from memory 
	Arguments	: Nothing
	Returns		: char c = binary data in the memory
*/
char CLoader::getchar()
{
	return *pGet++;
}

//	Private Methods _________________________________________________________
/*
	Method 		: extractData()
	Description	: Extract S1 and S2 records from S-Format file, 
				  and then convert them to binary data.
	Arguments	: Nothing
	Returns		: Nothing
*/
void CLoader::extractData()		
{
	int		asize;						// address size
	int		bc;							// byte count
	uint 	laddr;						// load address
	char	buf[64];					// buffer for getting 1 line
	char	*src;						// pointer to buffer
	char	*dest;						// pointer to memory (destination)

//cout << "Extract data ...\n" << flush;
	nSize = 0;
	while(!fin.eof()){
		fin.getline(buf,63);				// get 1 line datan buffer	
		src = &buf[1];						// src -> record type

// pick up S1 and S2 record
		switch(*src++){
			case '1': asize = 2;break; 		// 2 byte address
			case '2': asize = 3;break; 		// 3 byte address
			default: asize = 0; break;
		}

// convert data to binary
		if(asize){		
			bc = getByte(src) - asize - 1; 	//	get byte count
			src += 2;

	//	get load address
			laddr = getLaddr(src,asize);	// get load address

	//	check end of data
			if(nSize < (laddr + bc))
				nSize = laddr + bc;			// update end address

	//	write binary data into memory 
			src += asize * 2; 
			dest = pLM + laddr;  
			while(bc--){
				*dest++ = getByte(src); 
				src += 2;
			}
		}
	}
}

/*
	Method 		: getByte()
	Description	: get byte form HEX-ASCII string  
	Arguments	: int cnt 
	Returns		: int load address 
*/
char CLoader::getByte(char* ptr)
{
	return((hex2bin(*ptr) << 4) + hex2bin(*(ptr+1)));
}

/*
	Method 		: getLaddr()
	Description	: get load address
	Arguments	: int cnt 
	Returns		: int load address 
*/
uint CLoader::getLaddr(char* ptr, int cnt)
{
	uint	i,j;
	uint addr = 0;

	for(i = 0; i < cnt; i++){
		j = getByte(ptr) & 0xff;
		addr = (addr << 8) + j;
		ptr += 2;
	}
	if(nOffset)
		addr -= nOffset;
	return addr;
}

/*
	Method 		: hex2bin()
	Description	: Convert HEX-ASCII to Binary
	Arguments	: int n = HEX-ASCII code 
	Returns		: int binary code 
*/
int CLoader::hex2bin(int n)
{
	n -= '0';
	if(n > 9) n -= 7;
	return(n);
}

