/*
 	This file is part of Jenux.

    Jenux is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    Jenux is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with Jenux.  If not, see <http://www.gnu.org/licenses/>.

	Module			: CSerial.C
	Description		: Implementation of CSerial class. 
					  This class provides functionality for serial 
					  communications by MCU that works as UART. 
	Compiler		: gcc 4.8.5++
	Email			: matsuzawa.jei@nifty.com
	Maintenance		:
		Jul.13, '17	Released as GPL software officially
*/
#include <iostream>
#include <cstdio>
using namespace std;

#include <cstring>	
#include <queue>
#include <unistd.h>			
#include <fcntl.h>	
#include <sys/ioctl.h>	
#include <stdlib.h>

#include 	"../../../include/pifg/libPIF.h"
#include    "../../../include/pif/CSerial.h"
#include    "../../../include/pifg/CLog.h"

//	External classes ________________________________________________________
extern CLog	oLog;

std::map<int,std::string> CSerial::_ms;
std::map<int,int> CSerial::_mi;
std::map<int,int> CSerial::_mp;
std::map<int,int> CSerial::_mport;
int CSerial::_scratchStatus=0;
pthread_mutex_t	CSerial::_mutex_id;
pthread_mutex_t	CSerial::_mutex_CSerial;
pthread_mutex_t	CSerial::_mutex_CmdSeq;
int	CSerial::_seqMutexGo = 0;
int _lockNdx=0;

//#define _CSDEBUG_

void CSerial::concats(char *dst, char *src1, char *src2)
{
	//int totlen;
	if(dst)
	{
		if(src1) {
			if(src2) sprintf(dst,"%s:%s",src1,src2);
			else sprintf(dst,"%s",src1);
		} else {
			if(src2) sprintf(dst,"%s",src2);
			else *dst=0;
		}
	}
}

void CSerial::LockID()
{
	pthread_mutex_lock( &_mutex_id );
}
void CSerial::UnlockID()
{
	pthread_mutex_unlock( &_mutex_id );
}
int CSerial::GetLockID()
{
	static int lockID=0;
	int ret;
	
	LockID();
	lockID++;
	ret=lockID;
	UnlockID();
	
	return ret;
}

void CSerial::ShowIDs()
{
	std::map<int,string>::iterator msit;
	std::map<int,int>::iterator miit;
	std::map<int,int>::iterator mpit;
	std::map<int,int>::iterator mportit;
	
	LockID();
	printf("<<< LOCK IDs [%d]>>>\n",_scratchStatus);
	for(msit=_ms.begin(),miit=_mi.begin(),mpit=_mp.begin(),mportit=_mport.begin();msit != _ms.end();++msit,++miit,++mpit,++mportit)
	{
		printf("<<< port=%d, ids=%d, st=%d, puts=%d, %s >>>\n",mportit->second,msit->first,miit->second,mpit->second,msit->second.c_str());
	}
	UnlockID();
}

int CSerial::lock(char* s)
{
	int lid;
	char str[256];
	char dst[256];
	
	lid=GetLockID();
	sprintf(str,"CS::lock[%s][%d]",port?"isd":"eif",lid);
	if(s)
		concats(dst,s,str);
	else 
		strcpy(dst,str);
#ifdef _CSDEBUG_
	if(s) printf("<<< bef-%s >>>\n",dst);
#endif
	LockID();
	//if(s) _ms[lid]=s;
	///_ms[lid]=s;
	_ms[lid]=dst;
	//if(dst) _ms[lid]=dst;
	//else _ms[lid]="blank";
	_mi[lid]=1;
	_mp[lid]=1;
	_mport[lid]=port;
	UnlockID();
	pthread_mutex_lock( &_mutex_CSerial );
#ifdef _CSDEBUG_
	if(s) printf("<<< aft-%s >>>\n",dst);
#endif
	LockID();	// lock because others might be changing other lid in _mi
	_mi[lid]=2;
	_mp[lid]=2;
	UnlockID();
	return lid;
}

void CSerial::unlock(int lid, char* s)
{
	char str[256];
	char dst[256];
	
	sprintf(str,"CS::unlk[%s][%d]",port?"isd":"eif",lid);
	concats(dst,s,str);
#ifdef _CSDEBUG_
	if(s) printf("<<< %s >>>\n",dst);
#endif
	pthread_mutex_unlock( &_mutex_CSerial );
	LockID();
	_ms.erase(lid);
	_mi.erase(lid);
	_mp.erase(lid);
	_mport.erase(lid);
	UnlockID();
}

int CSerial::BeginCmdSeq(char *s)
{
	//int lockNdx=_lockNdx++;
	int lid;
	char str[256];
	char dst[256];
	
	lid=GetLockID();
	sprintf(str,"CS::BegCS[%s][%d]",port?"isd":"eif",lid);
	if(s)
		concats(dst,s,str);
	else
		strcpy(dst,str);
#ifdef _CSDEBUG_
	if(s) printf("<<< bef-%s >>>\n",dst);
#endif
	LockID();
	//if(s) _ms[lid]=s;
	//else _ms[lid]="blank";
	_ms[lid] = dst;
	_mi[lid]=3;		// before lock state
	UnlockID();
	pthread_mutex_lock( &_mutex_CmdSeq );
#ifdef _CSDEBUG_
	if(s) printf("<<< aft-%s >>>\n",dst);
#endif
	LockID();
	_mi[lid]=4;
	UnlockID();
	return lid;
}

void CSerial::EndCmdSeq(int lid, char *s)
{
	char str[256];
	char dst[256];
	
	sprintf(str,"CS::EndCS[%s][%d]",port?"isd":"eif",lid);
	concats(dst,s,str);
#ifdef _CSDEBUG_	
	if(s) printf("<<< %s >>>\n",dst);
#endif
	pthread_mutex_unlock( &_mutex_CmdSeq );
	LockID();
	_ms.erase(lid);
	_mi.erase(lid);
	UnlockID();
}
//	Global methos ___________________________________________________________
/*
	<method>		getchar	
	<description> 	RX a character	
					Returns a receiverd character
	<args>			none	
	<return>		int = c > 0 : Rx character 
						      0 : timeout
*/
int CSerial::getchar(const int offset, char *s)
{
	int c = 0;
    char dst[256];
	int lid;

    concats(dst,s,(char*)"getchar");
	lid=lock(dst);
    
	if(read(nFd,&c,1 | offset) > 0)	// RX a character
		c &= 0xff;
    unlock(lid,dst);
	return c;
}

/*
	<method>		isEmpty	
	<description> 	Check if RX buffer is empty	
					Returns RX buffer status
	<args>			none	
	<return>		int = true(1) : yes 
						  false(0): no 
*/
int	CSerial::isEmpty(char *s)
{
	int	ret = false;
 //   int lid;
	
	char dst[256];

//	printf("<<< in isEmpty %d >>>\n",cnt);
	concats(dst,s,(char*)"isEmpty");
//	lid=lock(dst);

	if(!port){						// EIF
		if(!ioctl(nFd,PIF_RD_EIFRXC))	// RX buffer empty ?
			ret = true;					//	- yes
	}
	else{							// O.P
		if(!ioctl(nFd,PIF_RD_OPRXC))	// RX buffer empty ?
			ret = true;					//	- yes
	}
//	unlock(lid,(char*)"isEmpty");
	return ret;
}

void CSerial::waitOPRX()
{
	int lid;
	static int cnt=0;

	printf("<<< in waitOPRX %d >>>\n",cnt);
	lid=lock((char*)"waitOPRX");
	//ioctl(nFd,PIF_WAIT_OPRX);
	unlock(lid,(char*)"waitOPRX");
	printf("<<< out waitOPRX %d >>>\n",cnt++);
}

/*
	<method>		putchar	
	<description> 	TX a character	
	<args>			char c = Tx character
	<return>		int = true(1)  : OK
						  false(0) : NG
*/
/*int CSerial::putchar(char c, char *s)
{
	int result;
	char dst[256];
	int lid;

	concats(dst,s,"putchar");
	lid=lock(dst);
	ioctl(nFd,PIF_WT_CHANNEL,port);
	result=write(nFd, &c, 1);
	unlock(lid,dst);
	return result;
}*/

/*
	<method>		puts	
	<description> 	TX character string terminated by NULL	
	<args>			char* ptr = pointer to string 
	<return>		int = true(1)  : OK
						  false(0) : NG
*/

void PrintStrToHex(const char *str)
{
	int i,l,m;
	
	l=strlen(str);
	for(i=0; i<l;i++)
	{
		printf("%X ",str[i]);
		m=i%16;
		if(15==(i%16)) {
			printf("\n");
		}
	}
	printf("\n");
}

//int CSerial::puts(const char* str, char *s)
int CSerial::puts(const char* str, const int offset, char *s)
{
	int	len;
	char dst[256];
	int lid;
    
    _scratchStatus=80;
	concats(dst,s,(char*)"puts");
	if( str ) {
		len = strlen(str);
		if(len>64)
		{
			printf("***EXCEEDED BUFSZ 64***\n");
			exit(0);
		}
	} else {
		return false;
	}
    _scratchStatus=81;
//	lid=lock(dst);
	_scratchStatus=82;
	LockID();
//	_mp[lid] = 3;
	UnlockID();
	LockID();
//	_mp[lid] = 4;
	UnlockID();
	_scratchStatus=83;
	while(len > 0){
		_scratchStatus=84;
		LockID();
//		_mp[lid] = 5;
		UnlockID();
		len = write(nFd, str, strlen(str) | offset);
		LockID();
//		_mp[lid] = 6;
		UnlockID();
		_scratchStatus=85;
		if(len > 0){
			str += (strlen(str) - len);	
		}
		_scratchStatus=86;
	}
	LockID();
//	_mp[lid] = 7;
	UnlockID();
	_scratchStatus=87;
//	unlock(lid,dst);
	_scratchStatus=88;
	
	if(!len)return true;
	else 	return false;
}

/*
	<method>		putn	
	<description> 	TX character string 	
	<args>			char* ptr = pointer to string 
					int n = string length
	<return>		int = true(1)  : OK
						  false(0) : NG
*/
int CSerial::putn(char* str, int n)
{
	char	c;
	int		len = n;
	int		tmp;	

	while(len > 0){
		len = write(nFd, str, n);
		if(len > 0){
			tmp = n - len;
			str += tmp;	
			n -= tmp;
			cout << "!" << flush;
		}
	}

	if(!len)return true;
	else 	return false;
}

/*
	<method>		getRegister
	<description>	Read PIF & PCI9054 local registers
					Returns a value of specified register
	<args>			int index = PIF register #
	<return>		uint = register value
*/
uint CSerial::getRegister(int index)
{
	REG	reg;
    uint result;
	int lid;

	lid=lock((char*)"getRegister");
	reg.offset = index;
	result=ioctl(nFd, PIF_RD_REG, &reg);
	unlock(lid,(char*)"getRegister");
	return result;
}

/*
	<method>		putRegister
	<description>	Write to PIF or PCI9050 local registers
					Writes a value into specified register
	<args>			int index = register #
					uint val = value to be written
	<return>		none
*/
void CSerial::putRegister(int index, uint val)
{
	REG	reg;
    int lid;

	lid=lock((char*)"putRegister");
	reg.offset = index;
	reg.val = val;
	ioctl(nFd, PIF_WT_REG, &reg);
    unlock(lid,(char*)"putRegister");
}

/*
	<method>		putBaud
	<description>	Set baud rate of channel-B (TDU)
	<args>			int baud = baud#
	<return>		none
*/
void CSerial::putBaud(int baud)
{
	if((baud < BAUD_9600) || (baud > BAUD_38400))
		baud = BAUD_19200;
	ioctl(nFd, PIF_WT_BAUD, baud);
}

/*
	<method>		putTrigger
	<description>	Set RX trigger level of channel-B (TDU)
	<args>			int trigger = trigger#
	<return>		none
*/
void CSerial::putTrigger(int trigger)
{
	if((trigger < TRIGGER_1) || (trigger > TRIGGER_14))
		trigger = TRIGGER_1;
	ioctl(nFd, PIF_WT_TRIGGER, trigger);
}


