////////////////////////////////////////////////////////////////////////////////
//	Module			: serial.C
//	Description		: Serial comunication handler
//
//	Compiler		: gcc version 3.2 (Red Hat Linux 8.0 3.2-7) 
//	Author			: T.Matsuzawa of JEI Matsumoto Lab.
//	E-mail			: tmatsu@po.mcci.or.jp
//------------------------------------------------------------------------------
//	Maintenance:
//		Apr. 9,'03	EV2.0	Ported on Redhat Linux 8 
//-----------------------------------------------------------------------------
////////////////////////////////////////////////////////////////////////////////
#include	<iostream>
#include 	<fstream>
using namespace std;

//#include 	<cstdlib>
//#include 	<cstring>
#include 	<unistd.h>
#include 	<fcntl.h>
#include 	<termios.h>
#include 	<sys/signal.h>
#include 	<sys/types.h>
#include 	<sys/sem.h>
#include 	<sys/ipc.h>
#include 	<sys/types.h>

#include 	"comserial.h"

#define COM1DEVICE			      "/dev/ttyS0"
#define COM2DEVICE			      "/dev/ttyS1"
#define _POSIX_SOURCE 1

//	Global Methods _____________________________________________________

int serial_init(int comdevice);

//	Local Methods ______________________________________________________

void serial1_IO_signal_handler(int status);    //signal handler for com1
void serial2_IO_signal_handler(int status);    //signal handler for com2
void serial_resetbuffer(int comdevice);
void serial_initbuffer(int comdevice);
void serial_freebuffer(int comdevice);
void serial_initsettings(int comdevice);
void serial1getavailabledata();
void serial2getavailabledata();
int serial1_getstring(char* buf, int nSize);
int serial2_getstring(char* buf, int nSize);
int getserialfiledesc(int comdevice);

// global variables
struct termios ser1_oldtio;
struct termios ser2_oldtio;

int ser1_portfd;
int ser2_portfd;

long  BAUD = B9600;
long  DATABITS = CS8;
long  STOPBITS = 0;
long  PARITYON = 0;
long  PARITY = 0; 	// 00 = NONE, 01 = Odd, 02 = Even, 03 = Mark, 04 = Space

#define BUF_SIZE	1024 
#define STRING_SIZE 8

char* buf1_array;
char* copy_buf1_array;
char* orig_buf1_array;
char* buf2_array;
char* copy_buf2_array;
char* orig_buf2_array;

int buf1_write_idx = 0;

int buf1_read_idx = 0;
int nUnreadChars1 = 0;
int isbusy1 = 0;

int buf2_write_idx = 0;
int buf2_read_idx = 0;
int nUnreadChars2 = 0;
int isbusy2 = 0;

/**********************************************************************
int serial_init()
arguments: comdevice - 1 = com1, 2 = com2
returns -1 on error and file descriptor of serial port on success
**********************************************************************/
int serial_init(int comdevice)
{
	struct sigaction saio;
	int portfd = -1;

	switch (comdevice) {
	// open the device(com port) to be non-blocking
    case 1:
	    ser1_portfd = open(COM1DEVICE, O_RDONLY | O_NOCTTY | O_NONBLOCK);
		portfd = ser1_portfd;
		break;
    case 2:
		ser2_portfd = open(COM2DEVICE, O_RDONLY | O_NOCTTY | O_NONBLOCK);
		portfd = ser2_portfd;
		break;
	default:
		return -1;
	}

	if (portfd != -1) {
		//install the serial handler before making the device asynchronous
		if (comdevice == 1)
			saio.sa_handler = serial1_IO_signal_handler;
		else
			saio.sa_handler = serial2_IO_signal_handler;    
		sigemptyset(&saio.sa_mask);
		saio.sa_flags = 0;
  		saio.sa_restorer = NULL;
  		sigaction(SIGIO,&saio,NULL);

		// allow the process to receive SIGIO
		fcntl(portfd, F_SETOWN, getpid());

		// Make the file descriptor asynchronous 
		fcntl(portfd, F_SETFL, FASYNC);

		if (comdevice == 1) 				// save current port settings
			tcgetattr(portfd, &ser1_oldtio);
    	else 								//initialize port's settings
			tcgetattr(portfd, &ser2_oldtio);

		serial_initsettings(comdevice);    
		serial_initbuffer(comdevice);
	}
	return portfd;
}

//	Local Methods _________________________________________________________

static void serial_initsettings(int comdevice)
{
	int portfd = getserialfiledesc(comdevice);
	struct termios ser_newtio;
  
	if(portfd != -1) {
		tcgetattr(portfd, &ser_newtio);
		ser_newtio.c_cflag |= 
			BAUD|CRTSCTS|DATABITS|STOPBITS|PARITYON|PARITY|CLOCAL|CREAD;
		ser_newtio.c_iflag = IGNPAR;

		//disable software flow control
		ser_newtio.c_iflag &= ~(IXON | IXOFF | IXANY);
		ser_newtio.c_oflag = 0;

		// for canonical input processing (raw data)			
		ser_newtio.c_lflag = 0;

		//enable read to return immediately even if no data is available
		ser_newtio.c_cc[VMIN]=0;
		ser_newtio.c_cc[VTIME]=0;
		tcsetattr(portfd, TCSANOW, &ser_newtio);
	}
}

void serial_initbuffer(int comdevice)
{
	if(comdevice == 1) {
		buf1_array = new char[BUF_SIZE];
		orig_buf1_array = copy_buf1_array = buf1_array;
		serial_resetbuffer(comdevice);
  	}
	else if (comdevice == 2) {
		buf2_array = new char[BUF_SIZE];
		orig_buf2_array = copy_buf2_array = buf2_array;
		serial_resetbuffer(comdevice);
	}
	else
    	cout << "unavailable com port\n";
}

void serial_resetbuffer(int comdevice)
{
	if (comdevice == 1) {
		buf1_write_idx = 0;
		buf1_read_idx = 0;
		buf1_array = copy_buf1_array = orig_buf1_array;	  
		 memset(orig_buf1_array, '\0', BUF_SIZE);
	}
	else {
		buf2_write_idx = 0;
		buf2_read_idx = 0;
		buf2_array = copy_buf2_array = orig_buf2_array;	  
		memset(orig_buf2_array, '\0', BUF_SIZE);
	}
}

void serial_freebuffer(int comdevice)
{
	if (comdevice == 1)
		delete[] orig_buf1_array;
	else if (comdevice == 2)
		delete[] orig_buf2_array;   
}

/*
void serial_setwordlength(int comdevice, unsigned char c)
{
  int portfd = -1;
	struct termios ser_newtio;
  
  portfd = getserialfiledesc(comdevice);

  if (portfd == -1)
		return;

  tcgetattr(portfd, &ser_newtio);
	ser_newtio.c_cflag &= ~CSIZE;  
	switch (c)
	{
		case 0:
			ser_newtio.c_cflag |= CS7;
			break;
		case 1:
			ser_newtio.c_cflag |= CS8;
			break;
		default: // set databits to CS8 by default
			ser_newtio.c_cflag |= CS8;
			break;
	}

  tcsetattr(portfd, TCSANOW, &ser_newtio);
}

void serial_setstopbits(int comdevice, unsigned char c)
{
  int portfd = -1;
	struct termios ser_newtio;

  portfd = getserialfiledesc(comdevice);
  if (portfd == -1)
		return;

  tcgetattr(portfd, &ser_newtio);
	switch (c)
	{
		case 0:
			ser_newtio.c_cflag &= ~CSTOPB;
			break;
		case 1:
		case 2:
			ser_newtio.c_cflag |= CSTOPB;
			break;
		default:
			ser_newtio.c_cflag &= ~CSTOPB;
			break;
			
	}
  tcsetattr(portfd, TCSANOW, &ser_newtio);
}

void serial_setparity(int comdevice, unsigned char c)
{
  int portfd = -1;
	struct termios ser_newtio;
//long  PARITY = 0;              // 00 = NONE, 01 = Odd, 02 = Even, 03 = Mark, 04 = Space  

  portfd = getserialfiledesc(comdevice);
  if (portfd == -1)
		return;

  tcgetattr(portfd, &ser_newtio);
	switch (c)
	{
		case 0: //none
			ser_newtio.c_cflag	&= ~PARODD;
			ser_newtio.c_cflag	&= ~PARENB;		
			break;
		case 1: //odd
			ser_newtio.c_cflag	|= PARENB;		
			ser_newtio.c_cflag	|= PARODD;
			break;
		case 2: //even
			ser_newtio.c_cflag	|= PARENB;		
			ser_newtio.c_cflag	&= ~PARODD;
			break;
		case 3: //mark (not supported in linux)
			//parityon = PAREXT;
			ser_newtio.c_cflag	&= ~PARENB;		
			ser_newtio.c_cflag	|= PARODD;
			break;
		case 4: //space (not supported in linux)
			//parityon = PAREXT;
			ser_newtio.c_cflag	&= ~PARODD;
			ser_newtio.c_cflag	&= ~PARENB;
			break;
		default: //set default parity to none
			ser_newtio.c_cflag	&= ~PARODD;
			ser_newtio.c_cflag	&= ~PARENB;
			break;
	}
  tcsetattr(portfd, TCSANOW, &ser_newtio);

}

void serial_setflowctrl(int comdevice, unsigned char c)
{
  int portfd = getserialfiledesc(comdevice);

	struct termios ser_newtio;

  if (portfd == -1)
		return;


  tcgetattr(portfd, &ser_newtio);
  switch (c)
  {
		case 0:
			ser_newtio.c_cflag &= ~CRTSCTS;
			ser_newtio.c_iflag |= (IXON | IXOFF);
			break;
		case 1:
		case 2: //etx = obsolete, use hardware flow control instead
		case 3:
		case 4:
		default: 			 /*set default to  XON XOFF */
			ser_newtio.c_cflag |= CRTSCTS;
			ser_newtio.c_iflag &= ~(IXON | IXOFF | IXANY);
			break;
	}

  tcsetattr(portfd, TCSANOW, &ser_newtio);  
}
*/

int getserialfiledesc(int comdevice)
{
	if (comdevice == 1)
		return ser1_portfd;
	else if (comdevice == 2)
		return ser2_portfd;
	return -1;
}

/*******************
int serial_setbaudrate (int nBaudRate)
ARGUMENT(S): nBaudRate
- Baud Rate of serial

********************/
/*
void serial_setbaudrate(int comdevice, int nBaudRate)
{
  int portfd = getserialfiledesc(comdevice);
  speed_t speed = BAUD;  
  struct termios ser_newtio;

  switch (nBaudRate)
  {
		case 0:
			speed = B300;
			break;
		case 1:
			speed = B1200;
			break;
		case 2:
			speed = B2400;
			break;
		case 3:
			speed = B4800;
			break;
		case 4:
			speed = B9600;
			break;
		case 5:
      speed = B19200;
			break;
		case 6:
			speed = B38400;
			break;
		case 7:
			speed = B57600;
			break;
		case 8:
			speed = B115200;
			break;
		default:
			break;
	}

  if (portfd != -1)
  {
      tcgetattr(portfd, &ser_newtio);
      ser_newtio.c_cflag &= ~CBAUD;
      ser_newtio.c_cflag |= speed;
  		tcsetattr(portfd, TCSANOW, &ser_newtio);
	}

}
*/

/******************************
char serial_getchar()
returns null terminator on error
*******************************/
unsigned char serial_getchar(int comdevice)
{
  char c = '\0';
  if (comdevice == 1)
  {
    if (buf1_read_idx == BUF_SIZE)
	  	 buf1_read_idx = 0;
   	c = copy_buf1_array[buf1_read_idx++];
	  nUnreadChars1--;
    if (nUnreadChars1 == 0)
       serial1getavailabledata();
  }
  else if (comdevice == 2)
  {
    if (buf2_read_idx == BUF_SIZE)
		  buf2_read_idx = 0;

	  c = copy_buf2_array[buf2_read_idx++];
	  nUnreadChars2--;
    if (nUnreadChars2 == 0)
       serial2getavailabledata();
	  
  }
  return c;
}

/******************************
char serial_getstring()
returns number of bytes read
*******************************/
int serial1_getstring(char* buf, int nSize)
{
  if (NULL != buf)
     return read(ser1_portfd, buf, nSize);
    
  return 0;
}


int serial2_getstring(char* buf, int nSize)
{
  if (NULL != buf)
     return read(ser2_portfd, buf, nSize);

  return 0;
}

/********************************
returns 0 on success, -1 on error
*********************************/
int serial_close(int comdevice)
{
  int nRet = -1;
  
  if (comdevice == 1)
  {
    if (ser1_portfd != -1)
    {
  	  // restore old port settings
	    tcsetattr(ser1_portfd, TCSANOW, &ser1_oldtio);
      //close the com port
 	    nRet = close(ser1_portfd);
    }
  }
  else if (comdevice == 2)
  {
    if (ser2_portfd != -1)
    {
  	  tcsetattr(ser2_portfd, TCSANOW, &ser2_oldtio);
      //close the com port
 	    nRet = close(ser2_portfd);
    }
  }
  
  serial_freebuffer(comdevice);
  return nRet;
}

/***************************************************************************
* sets wait_flag to FALSE, to indicate that characters have been received.                                           *
***************************************************************************/
void serial1_IO_signal_handler (int status)
{
  //if previous set of data has been read, reset buffer
  sleep(1);
  if (nUnreadChars1 == 0)
  {
  	serial_resetbuffer(1);
    isbusy1 = false;
  }
  else if (isbusy1)
  {
    return;
  }

  serial1getavailabledata();
}

void serial1getavailabledata()
{
  char temp[STRING_SIZE];
  int nCount = STRING_SIZE;

  isbusy1 = true;
  while (nCount > 0)
  {
		//stop receiving data, resume only when nUnreadChars is empty
	if ((nUnreadChars1 + STRING_SIZE) > BUF_SIZE)
	{
		isbusy1 = false;  
		return;
	}
		
	nCount = serial1_getstring(temp, STRING_SIZE);
    if (nCount == 0)
    {
      isbusy1 = false;  
      return;
    }
    if (nCount < STRING_SIZE)
		temp[nCount] = '\0';

    //write only the data that will fit in the buffer
    //and then go to the start of the array again
    if ((nCount + buf1_write_idx) > BUF_SIZE)
	{
		int nWrite = BUF_SIZE - buf1_write_idx;
		char* buf = NULL;
		char* tempbuf = NULL;

		strncpy(buf1_array, temp, nWrite);
		nUnreadChars1 += nWrite;
		buf1_array = orig_buf1_array;
		buf1_write_idx = 0;

		buf = new char[sizeof(temp)];
		memset(buf, '\0', sizeof(buf));
		strcpy(buf, temp);
		tempbuf = buf;
		buf += nWrite;
		memset(temp, '\0', sizeof(temp));
		strcpy(temp, buf);
		delete[] tempbuf;
		nCount -= nWrite;
	}

    strcpy(buf1_array, temp);
	buf1_array += nCount;
	buf1_write_idx += nCount;
  //for debugging purposes only
 	// printf("%d\n", buf1_write_idx);
	nUnreadChars1 += nCount;
	if (buf1_write_idx == BUF_SIZE)
    {
		buf1_write_idx = 0;
		buf1_array -= BUF_SIZE;
		break;
    }
  }
  isbusy1 = false;  
}

void serial2_IO_signal_handler (int status)
{
  sleep(1);
  //if previous set of data has been read, reset buffer
  if (nUnreadChars2 == 0)
  {
  	serial_resetbuffer(2);
    isbusy2 = false;
  }
  else if (isbusy2)
  {
    return;
  }

  serial2getavailabledata();
}

void serial2getavailabledata()
{
  char temp[STRING_SIZE];
  int nCount = STRING_SIZE;

  isbusy2 = true;
  while (nCount > 0)
  {
	//stop receiving data, resume only when nUnreadChars is empty
	if ((nUnreadChars2 + STRING_SIZE) > BUF_SIZE)
	{
		isbusy2 = false;
		return;
	}
		
	nCount = serial2_getstring(temp, STRING_SIZE);
    if (nCount == 0)
    {
       isbusy2 = false;
       return;
    }
     
    if (nCount < STRING_SIZE)
		temp[nCount] = '\0';

    //write only the data that will fit in the buffer
    //and then go to the start of the array again
    if ((nCount + buf2_write_idx) > BUF_SIZE)
	{
		int nWrite = BUF_SIZE - buf2_write_idx;
		char* buf = NULL;
	    char* tempbuf = NULL;

		strncpy(buf2_array, temp, nWrite);
		nUnreadChars2 += nWrite;
		buf2_array = orig_buf2_array;
		buf2_write_idx = 0;

		buf = new char[sizeof(temp)];
		memset(buf, '\0', sizeof(buf));
		strcpy(buf, temp);
		tempbuf = buf;
		buf += nWrite;
		memset(temp, '\0', sizeof(temp));
		strcpy(temp, buf);
	    delete[] tempbuf;
		nCount -= nWrite;
	}

    strcpy(buf2_array, temp);
	buf2_array += nCount;
	buf2_write_idx += nCount;
  //for debugging purposes only
 	// printf("%d\n", buf2_write_idx);
	nUnreadChars2 += nCount;
	if (buf2_write_idx == BUF_SIZE)
    {
		buf2_write_idx = 0;
		buf2_array -= BUF_SIZE;
		break;
    }
  }

  isbusy2 = false;
}

int serial_getlength(int ncomdevice)
{
  if (ncomdevice == 1)
    return nUnreadChars1;
  else if (ncomdevice == 2)
    return nUnreadChars2;
  return 0;
}

unsigned long serial_ungetchar(int ncomdevice, unsigned char c)
{
  if (ncomdevice == 1)
  {
    if (copy_buf1_array[(buf1_read_idx - 1)] == c)
    {
  	  buf1_read_idx--;
      nUnreadChars1++;
      return 0;
    }
  }
  else if (ncomdevice == 2)
  {
    if (copy_buf2_array[(buf2_read_idx - 1)] == c)
    {
  	  buf2_read_idx--;
      nUnreadChars2++;
      return 0;
    }
  }
  return 1;
}

unsigned char serial_peekc(int comdevice)
{
 unsigned char c = '\0';
 
 if (comdevice == 1)
		c = copy_buf1_array[buf1_read_idx];
  else if (comdevice == 2)
		c = copy_buf1_array[buf1_read_idx];
  return c;
  
}
