/*
 	This file is part of Jenux.

    Jenux is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    Jenux is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with Jenux.  If not, see <http://www.gnu.org/licenses/>.

	Module			: CPif.C
	Description		: Implementation of CPif class. This is a main class of
					  shared library libPIF.
	Compiler		: gcc 4.8.5++
	Email			: matsuzawa.jei@nifty.com
	Maintenance		:
		Jul.13, '17	Released as GPL software officially
*/
#include 	<iostream>
using namespace std;

#include	<stdio.h>
#include	<stdlib.h>
#include    <unistd.h>
#include    <sys/time.h>
#include    <sys/types.h>

#include 	"../../../include/pifg/libPIF.h"
#include <pthread.h>
#include <../../home/Genesis/include/lis.h>
#include "../../../include/pif/CTDU.h"

#define	DEBUG	1

//	External variables ----------------------------------------------------
CLog	oLog;
extern  GLOBALAREA*	   	globalarea;


//	Global functions ______________________________________________________

int	getTime();					// get current time in sec.

//	Local variables _______________________________________________________

pthread_mutex_t     mutex_eif;
pthread_mutex_t     mutex_pnl;
const int	hctbl[] =	{R_CBC, R_DPC, R_PCC, R_DUC };

//	Global functions ________________________________________________________
/*
	<method>		getTime
	<description>	Get current time in sec.
	<args>			none
	<return>		int = current time	
*/
int	getTime()
{
	struct	timeval		tv;
	struct	timezone	tz;
	
	gettimeofday(&tv, &tz);		// get current time
	return((int)tv.tv_sec);		// return sec.	
}

//	Global methods __________________________________________________________
/*
	<method>		CPif
	<description>	Constructor 
					- create CPrt class 
					- create CEif class 
					- create CPanal class 
					- create CList classes  
					- reset printer status
					- create wait_end Pthread
					- enable PWI (/PWAR interrupt)
	<args>			None
*/
CPif::CPif()
{
	int	fd;
	int	st,model;

	LOG("libPIF",LogInfo,"Creating CPif class instance ...");
	version = "3.2.1 (Sep.6, 2017)";

// create CPif class
	pPrt = new CPrt();
	if(pPrt->getFailed()){
		LOG("libPIF",LogErr,"Failed to create CPrt class.");
//		exit(1);
		fd = 0;
	}else
        fd = pPrt->getFd();
   if (fd){ 
	// create CEif class
	pEif = new CEif(fd);
	if(pEif < 0){
		LOG("libPIF",LogErr,"Failed to create CPanel class.");
	//	exit(1);
	}
	while(1){
        if(pPrt->getStatusRegister() & EACT){
            pEif->setEACT(true);
            break;
            
        }
        else{
            //return;
            static int failCount = 0;
            printf("EACT fail [%d]\n",failCount++);
            sleep(1);
        }
    }

	pPrt->enableINT(MUIA);				// enable UART intr.
	st = pEif->putRequest(R_PRI,-1,0);
	if(st < 0){
		LOG("libPIF",LogErr,"Failed to communicate with EIF!");
		exit(1);
	}
	cout << "PRI: " << st << endl;
	switch(st){
	case 10:
    	case 5:
		model = 0;
		break;
    case 14:
    case 4:
		model = 1;
		break;
	default:
		LOG("libPIF",LogErr,"EIF returned wrong PRI!");
		exit(1);
		break;
	}
	if(!model)
		LOG("libPIF",LogInfo,"GOP-5084 is selected.");
	else
		LOG("libPIF",LogInfo,"ISD-202 is selected.");
	pPrt->enableINT(MUIA+MUIB);				// enable UART intr.
	pPanel = new CTDU(fd,model,0);
	if(pPanel < 0){
		LOG("libPIF",LogErr,"Failed to create CTDU class.");
		exit(1);
	}
	pPanel->setEncode(pEif->getLANG());

//	get engine parameter
	for(int i = 0; i < EP_MAX; i++){
		enginePar.push_back(pEif->putRequest(R_VOR+i,-1,0));
	}

//	enable interrupts from Engine
	pPrt->enableINT(MUIB+MUIA+MERR+MWAR+MPWR);	// enable /PWAR+/PERR+/PWR intr.
	enableDBG();					// enable debug messages
    }
}

void CPif::LockISDWorker()
{
	pthread_mutex_lock(&_ISDWorkerMutex);
}

void CPif::UnlockISDWorker()
{
	pthread_mutex_unlock(&_ISDWorkerMutex);
}

/* 
 * desc:	For Synchronization purposes between threads.
 * 			When this function is called it will wait until 
 * 			it receives a signal from SignalISDWorker that 
 * 			it can continue.
 */
void CPif::WaitISDWorker(int withLock)
{
	char msg[256];

	if(withLock) LockISDWorker();
	//while(!_cancelISD && (1/* no key */))
	//while(!_cancelISD)
	{
		_ISDWorkerWait = 1;
		pthread_cond_wait(&_ISDWorkerCond,&_ISDWorkerMutex);
		_ISDWorkerWait = 0;
	}
	if(withLock) UnlockISDWorker();
}

/* 
 * desc:	For Synchronization purposes between threads.
 * 			When this function is called it signals the thread
 *			that called the WaitISDWorker to continue.
 */
void CPif::SignalISDWorker(int withLock)
{
	if(withLock) LockISDWorker();
	if(_cancelISD)
	{
		pthread_cond_signal(&_ISDWorkerCond);
	} 
	else if(_ISDWorkerWait)
	{
		pthread_cond_signal(&_ISDWorkerCond);
	} else {
		//printf("SIW skip\n");
	}
	if(withLock) UnlockISDWorker();
}

/* void ISDReceiverThreading(...)
 * input:	none
 * output:	none
 * desc:	This is a separately running thread that
 * 			handles the receiver of key inputs from the panel
 */
void CPif::ISDReceiverThreading()
{
	int result;
	char str[256];
	static int it=0;
//   	GOPST st;
	printf("Entered ISDReceiverT\n");
	_ISDReceiverThreadUp=1;
	pPanel->clearScreen();
	while(!_cancelISD)
	{
		usleep(1);
		// Wait and get next key input from panel
		result = p_getKey(st);
		printf("it=%d\n",it++);
		sprintf(str,"type_=%d, cmd_=%d, par_=%d\n",st.type,st.cmd,st.par);
		printf("<<<%s>>>\n", str );
		printf("Before if(!_can...res = %d\n",result);
		if(!_cancelISD)
		{
			printf("if(!_can...\n");
			if(result)
			{
				SignalISDWorker(1);
				printf("After SignalISDWorker(1)\n");
#if 0
				switch(st.cmd) {
				case 'M':
					if( screen == 5 ) {
						printf("going 6 fr 5\n");
						//globalarea->Bib->OP_ClearScreen(lib);
						globalarea->Bib->OP_DispPage(lib,6);
						//globalarea->Bib->OP_DispPage(lib,5);
						screen = 6;
					} else if( screen == 6 ) {
						printf("going 5 fr 6\n");
						//globalarea->Bib->OP_ClearScreen(lib);
						globalarea->Bib->OP_DispPage(lib,5);
						screen = 5;
					}
					break;
				case 'H':
					printf("going 5 fr 7\n");
					//globalarea->Bib->OP_ClearScreen(lib);
					globalarea->Bib->OP_DispPage(lib,5);
					screen = 5;
					break;
				case 'R':
					printf("going 7\n");
					//globalarea->Bib->OP_ClearScreen(lib);
					globalarea->Bib->OP_DispPage(lib,7);
					//globalarea->Bib->OP_DispPage(lib,15);
					screen = 7;
					break;
				case 'B':
					printf("going 6 fr 7\n");
					//globalarea->Bib->OP_ClearScreen(lib);
					globalarea->Bib->OP_DispPage(lib,6);
					screen = 6;
					break;
				case 'E':
					//_iISD_Running = 0;
					break;
				case 'T':
					globalarea->Bib->OP_Disable(lib);
					sleep(5);
					globalarea->Bib->OP_Enable(lib);
					break;
				default:
					printf("ELSE\n");
					break;
				}
#else
				if( st.type == 'A' ) {
					
				} else {
				
				}
#endif
			}
		}
	}
	printf("Exit ISDReceiverT\n");
	_ISDReceiverThreadUp=0;
}

/* void ISDWorkerThreading(...)
 * input:	none
 * output:	none
 * desc:	This is a separately running thread that
 * 			handles the worker of translating key inputs into some form of output
 */
void CPif::ISDWorkerThreading()
{
	char msg[256];
	int key;
	
	printf("Entered ISDWorkerT\n");
	_ISDWorkerThreadUp=1;
	while(!_cancelISD)
	{
		WaitISDWorker(1);
		printf("After WaitISDWorker()\n");
		if(!_cancelISD) { 		// dual cancel checking necessary
			key = st.cmd;		//save current key for it can change in receiver thread while this thread is not done processing
			DoISDWorker(&key);
		}
	}
	printf("Exit ISDWorkerT\n");
	_ISDWorkerThreadUp=0;
}

/* void ISDReceiverThreadingStarter(...)
 * input:	void *arg - the calling object pointer
 * output:	none
 * desc:	Creates a normalized entry point for the 
 * 			actual thread "ISDReceiverThreading()"
 */
void CPif::ISDReceiverThreadingStarter(void *arg)
{
	CPif *pPif = (CPif*)arg;
	pPif->ISDReceiverThreading();
}

/* void StartISDReceiverThread(...)
 * input:	none
 * output:	none
 * desc:	initiates the thread "ISDReceiverThreading()"
 */
void CPif::StartISDReceiverThread()
{
	if( (retVal_ISDReceiverThread = pthread_create( &net_ISDReceiverThread, NULL, (void*(*)(void*))ISDReceiverThreadingStarter, (void*)this )) ) {
		perror( "ISD Receiver pthread_create()" );
		exit( 1 );
	}
}

/* void ISDWorkerThreadingStarter(...)
 * input:	void *arg - the calling object pointer
 * output:	none
 * desc:	Creates a normalized entry point for the 
 * 			actual thread "ISDWorkerThreading()"
 */
void CPif::ISDWorkerThreadingStarter(void *arg)
{
	CPif *pPif = (CPif*)arg;
	pPif->ISDWorkerThreading();
}

/* void StartISDWorkerThread(...)
 * input:	none
 * output:	none
 * desc:	initiates the thread "ISDWorkerThreading()"
 */
void CPif::StartISDWorkerThread()
{
	if( (retVal_ISDWorkerThread = pthread_create( &net_ISDWorkerThread, NULL, (void*(*)(void*))ISDWorkerThreadingStarter, (void*)this )) ) {
		perror( "ISD Worker pthread_create()" );
		exit( 1 );
	}
}

/* void AssignISDWorkerCallback()
 * input:	void (*ISDWorkerCallback)(char *)
 * output:	none
 * desc:	Pass the ISD received key code to the worker callback specified.
 */
void CPif::AssignISDWorkerCallback(void (*pISDWorkerCallback)(int *))
{
	LockISDWorker();
	_ISDWorkerCallback = pISDWorkerCallback;
	UnlockISDWorker();
}

/* void DoISDWorker()
 * input:	char* key
 * output:	none
 * desc:	Relays the input key code from panel to the ISD worker callback function if exist.
 */
void CPif::DoISDWorker(int* key)
{
	//LockISDWorker();
	if(_ISDWorkerCallback)
	{
		_ISDWorkerCallback(key);
	}
	//UnlockISDWorker();
}

/*
	<method>		~CPif
	<description>	Destructor 
					- delete all classes generated 
	<args>			none
*/
CPif::~CPif()
{
	delete pPrt;
	delete pEif;
	delete pPanel;
}

int CPif::SetDummy0()
{
	int z;
	
	z = 0;
	
	return z;
}

int CPif::SetDummy1(int a)
{
	int z;
	
	z = a;
	
	return z;
}

int CPif::SetDummy2(int a, int b)
{
	int z;
	
	z = a + b;
	
	return z;
}
/*
	<method>		getPrinterStatus	
	<description>	Get printer status
	<args>			none
	<return>		int 
*/
int CPif::getPrinterStatus(int iPST)
{
	static int cc = 0;

	//if(pEif->getEACT()){
		if( iPST < 0 ) {
			// just for reading
		} else if( iPST ) {
			nPST = iPST;
		} else {
			_PERR = pPrt->getPERR();
			_PWAR = pPrt->getPWAR();
			//if( _PWAR || _PERR || !nPST ) {
			//if( _PWAR || _PERR ) {
				nPST = pEif->getEST();
				if( (nPST & 0xff0000) == 0xff0000 ) {
					printf( "[ERRONEOUS ERROR!(%d) <%x>\n", cc++, nPST );
					sleep(1); ///
				}
				if( _PERR ) {
					//pPrt->resetPERR();
				}
				if( _PWAR ) {
					//pPrt->resetPWAR();
				}
			//}
		}
		nPST |= PST_POWER;
	//}
	return(nPST);
}

unsigned int CPif::setImageSize()
{
	int r,w,h,tm,bm,lm;

	printf("<<<GOING IN: CPif::setImageSize()>>>\n");
	r = res[e_getResolution(0)];
	cout << "eif res: " << r << endl;
	w = e_getFormWidth(0);
	cout << "eif form width: " << w << endl;
	h = e_getFormHeight(0);
	cout << "eif form height: " << h << endl;
	tm = e_getTopMargin(0);
	cout << "eif Top Margin: " << tm << endl;
	bm = e_getBottomMargin(0);
	cout << "eif Bottom Margin: " << bm << endl;
	lm = e_getLeftMargin(0);
	cout << "eif Left Margin: " << lm << endl;

	nWidth = ((w - lm - 10) * (r / 10) + 31) / 32;
	printf("<<<:setImageSize---nWidth = %d>>>\n", nWidth );
	nHeight = ((h - tm - bm) * r) / 6;
	printf("<<<:setImageSize---nHeight = %d>>>\n", nHeight );
	cout << "nWidth=" << nWidth << endl;
	cout << "nHeight=" << nHeight << endl;

	if(!tm && !bm)
		nHeight -= 1;				// minimum deadzone

	putDimReg(nHeight << 16 | nWidth);
	putSizeReg(nWidth * nHeight);

	nSize = nWidth * nHeight;
	printf("<<<GOING OUT: CPif::setImageSize()>>>\n");
	
	return nSize;
}

int CPif::checkPageBank()
{
  int bankSt;
  int ret;
  
  ret = 0;
  bankSt = getBufferStatus();
 
  if( (bankSt & BANK_A_READY) == BANK_A_READY )
  {
    printf("BANK_A_READY\n");
    ret |=1;
  }
  
  if( (bankSt & BANK_B_READY) == BANK_B_READY )
  {
     printf("BANK_B_READY\n");
    ret |=2;
  }
  return ret;
}

/*
	<method>		printPage
	<description>	Print a page (enter a page in waiting queue)	
	<args>			CFrameBuffer* page = pointer to a page
	<return>		int 
*/
int CPif::printPage(uint32_t* page)
{
	int	ret = true;
	
	return(ret);
}

/*
	<method>		getEACT
	<description>	returns EACT signal 
	<args>			none
	<return>		int true(1) = active
						false(0) = inactive
*/
int
CPif::getEACT()
{
	int	st;

	if(pPrt->getStatusRegister() & EACT)
		st = true;
	else
		st = false; 
	pEif->setEACT(st);
	return(st);
}

/*
	<method>		e_getHardCounter
	<description>	returns assembly counters
	<args>			none
	<return>		vector = vector of assembly counters
*/
vector<int>
CPif::e_getHardCounter()
{
	hardCounter.clear();

	for(int i = 0; i < HC_MAX; i++)
		hardCounter.push_back(pEif->putRequest(hctbl[i],-1,0));
	return hardCounter;
}

/*
	<method>		e_getSoftCounter
	<description>	returns software counters
	<args>			none
	<return>		vector = vector of software counters
*/
vector<int>
CPif::e_getSoftCounter()
{
	int	i;
	softCounter.clear();

	for(i = 0; i < LC_MAX; i++)
		softCounter.push_back(pEif->getLifeCounter(i));
	for(i = 0; i < PMC_MAX; i++)
		softCounter.push_back(pEif->getPMCounter(i));
	return softCounter;
}
