/*
 	This file is part of Jenux.

    Jenux is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    Jenux is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with Jenux.  If not, see <http://www.gnu.org/licenses/>.

	Module			: CPrt.C
	Description		: Implementation of CPrt class. 
					  This class provides functionality to access the pifc 
					  device. 
	Compiler		: gcc 4.8.5++
	Email			: matsuzawa.jei@nifty.com
	Maintenance		:
		Jul.13, '17	Released as GPL software officially
*/
#include <iostream>
#include <sstream>
using namespace std;

#include <string>
#include <cstring>
#include <fstream>
#include <sstream>
#include <stdlib.h>
#include <unistd.h>
#include <fcntl.h>
#include <errno.h>
#include <sys/ioctl.h>
#include <sys/mman.h>
#include <sys/wait.h>

#include 	"../../../include/pifg/libPIF.h"
#include <stdio.h>
#include <bits/string.h>
#include <string.h>
#include <linux/string.h>
//#include "pif/CPrt.h"
//#include 	"pifg/CLog.h"

//	External classes ________________________________________________________
extern CLog	oLog;

//	Local variables _________________________________________________________

const string	devfile = "/dev/pifII";		// device file name 

//	Public methods __________________________________________________________
/*
	<method>		CPrt
	<description>	Constructor
					Opens a pifc device 
	<args>			none
*/
CPrt::CPrt()
{
	ostringstream	os;

	LOG("libPIF",LogInfo,"Creating CPrf class instance ...");
// opens a pifc device

	cout << "start prt" << endl;
    	nFd = open("/dev/pifII", O_RDWR);
	printf("<<< :CPrt() nFd = %d >>>\n", nFd);
	if(nFd < 0){
		cout << "nFd failed" << endl;
		os << "Failed to open pifc device driver: " << strerror(errno);
		LOG("libPIF",LogErr,os.str());
		failed = 1;
//		exit(1);
	} else {
		cout << "nFd success" << endl;
	}
	pFB = (uint*)0;
//	re-map FIFO port to user address space
	pFIFO = (uint*)mmap( (void*)0, 1024*1024*32,PROT_WRITE|PROT_READ, 
					MAP_SHARED, nFd, 0);

//	putControlRegister(getControlRegister() & ~IBS);
	unsigned ver = ioctl(nFd, PIF_RD_VER);
	os.str("");
	os << (ver >> 8) << "." << ((ver >> 4) & 0xf) << "." << (ver & 0xf); 	
	DDversion = os.str();
	os.str("");
	os << "DD version: " << DDversion;
	LOG("libPIF",LogInfo,os.str());
}

/*
	<method>		~CPrt
	<description>	Destructor 
					Closes a pifc device 
	<args>			none
*/
CPrt::~CPrt()
{
	close(nFd);							// close a device
	cout << "Close a pifc device\n";
}

/*
	<method>		disableINT
	<description>	Disable LINTi1 interrupts
					Resets interrupr bits in the PIF Controll register
	<args>			int intr = interrupts	
	<return>		none
*/
void CPrt::disableINT(int intr)
{
	putControlRegister(getControlRegister() | intr);
}

/*
	<method>		enableINT
	<description>	Enable LINTi1 interrupts
					Sets interrupr bits in the PIF Controll register
	<args>			int intr = interrupts	
	<return>		none
*/
void CPrt::enableINT(int intr)
{
    putControlRegister(getControlRegister() & ~intr);
}

/*
	<method>		getFifoCounter
	<description> 	Get input fifo counter 
					Returns a value of 
	<args>			none	
	<return>		int = fifo counter 
*/
uint CPrt::getFifoCounter()
{
	return(*pFIFO);
}

/*
	<method>		getBufferStatus
	<description> 	Get image buffer status 
					Returns a value of image buffer register
	<args>			none	
	<return>		int = image buffer status
*/
uint CPrt::getBufferStatus()
{
	return(*(pFIFO+1) & 0x1ff);
}

/*
	<method>		getIntCnt
	<description>	Get number of interrupts
					Returns number of specified interrupt
	<args>			int index = interrupt #
								0: LINTi1 (total)
								1: PEI (/PERR)
								2: PWI (/PWAR)
	<return>		int = number of interrupts 
*/
int CPrt::getIntCnt(int index)
{
	return(ioctl(nFd, PIF_RD_INTC, index));
}

/*
	<method>		getIRQ
	<description>	Get IRQ for PIFC
					Returns IRQ #
	<args>			none
	<return>		int = IRQ #
*/
int CPrt::getIRQ()
{
	return(ioctl(nFd, PIF_RD_IRQ));
}

/*
	<method>		getPERR
	<description>	Get PEI flag 
					Returns a /PERR interrupt flag
	<args>			none
	<return>		int = flag	false(0) : no interrupt 
								true(1)	 : accepted interrupt 
*/
int	CPrt::getPERR()
{
	int ret;

	ret = ioctl(nFd, PIF_RD_PERR);

	return ret;
}

/*
	<method>		getPWAR
	<description>	Get PWI flag
					Returns a /PWAR interrupt flag
	<args>			none
	<return>		int = flag	false(0) : no interrupt
								true(1)	 : accepted interrupt 
*/
int	CPrt::getPWAR()
{
	int ret;
	
	ret = ioctl(nFd, PIF_RD_PWAR);

	return ret;
}

/*
	<method>		getRegister
	<description>	Read PIF & PCI9054 local registers
					Returns a value of specified register
	<args>			int index = PIF register #
	<return>		uint = register value
*/
uint CPrt::getRegister(int index)
{
	REG reg;

	reg.offset = index;
	return (ioctl(nFd, PIF_RD_REG, &reg));
}

/*
	<method>		putRegister
	<description>	Write to PIF or PCI9050 local registers
					Writes a value into specified register
	<args>			int index = register #
					uint val = value to be written
	<return>		none
*/
void CPrt::putRegister(int index, uint val)
{
	REG reg;

	reg.offset = index;
	reg.val = val;
	ioctl(nFd, PIF_WT_REG, &reg);
}
/*
	<method>		putImageData
	<description>	Write image data to buffer on the PIFC
	<args>			uint* ptr = pointer to video data 
					uint size = data size 
	<return>		none
*/
int CPrt::putImageData(uint *ptr, uint size)
{
#if	DMA_ENABLE
	uint	par[2];

	par[0] = (uint)(ptr - pFB) * 4;		// set start offset
	par[1] = size;						// set data size
	return(ioctl(nFd, PIF_WT_VIDEO, par));
#else
	uint*	d;

	d = pFIFO;
	for(int i = 0; i < size; i++)
		*d++ = *ptr++;
	return true;
#endif
}

/*
	<method>		ResetUart
	<description>	Reset Uart on the PIF 
	<args>			none
	<return>		none
*/
void CPrt::resetUart()
{
	ioctl(nFd, PIF_RST_UART);
}

/*
	<method>		resetPERR
	<description>	Reset a /PERR interrupt flag
	<args>			none
	<return>		none
*/
void CPrt::resetPERR()
{
	ioctl(nFd, PIF_RST_PERR);
}

/*
	<method>		resetPWAR
	<description>	Reset a /PWAR interrupt flag
	<args>			none
	<return>		none
*/
void CPrt::resetPWAR()
{
	ioctl(nFd, PIF_RST_PWAR);
}

/*
	<method>		ResetPif
	<description>	Reset PIF
	<args>			none
	<return>		none
	<note>			All registers in the PIF are reset.	
*/
void CPrt::resetPif()
{
	ioctl(nFd, PIF_RST_PIF);
}

/*
	<method>		ResetVideo
	<description>	Reset Video circuit
	<args>			none
	<return>		none
	<note>			All registers in the pIF are reset.	
*/
void CPrt::resetVideo()
{
	ioctl(nFd, PIF_RST_VIDEO);
}

/*
	<method>		waitVDRQ
	<description>	wait for /VDRQ interrupt
	<args>			none
	<return>		none
*/
int	CPrt::waitVDRQ()
{
    return(ioctl(nFd, PIF_RD_VDRQ));
}

//	Private methods ________________________________________________________

// disable bit swap
void
CPrt::disableBitSwap()
{
	uint32_t val;
	val = getRegister(R_PIF_CTRL);
	putRegister(R_PIF_CTRL,val & ~IBS);
}

// enable bit swap
void
CPrt::enableBitSwap()
{
	uint32_t val;
	val = getRegister(R_PIF_CTRL);
	putRegister(R_PIF_CTRL,val|IBS);
}

