/*
 	This file is part of Jenux.

    Jenux is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    Jenux is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with Jenux.  If not, see <http://www.gnu.org/licenses/>.

	Module			: CEif.C
	Description		: Implementation of CEif class. 
                      This class is a delived from CSerial class and
                      provides functionality to communicate with EIF.
	Compiler		: gcc 4.8.5++
	Email			: matsuzawa.jei@nifty.com
	Maintenance		:
		Jul.13, '17	Released as GPL software officially
*/
#include 	<iostream>
#include 	<iomanip>
#include 	<sstream>
using namespace std;

#include    <sys/ioctl.h>
#include    <unistd.h>
#include    <pthread.h>
#include	<stdio.h>
#include	<stdlib.h>

#include 	"../../../include/pifg/libPIF.h"
#include 	"../../../include/pif/CEif.h"
#include 	"../../../include/pifg/CLog.h"
#include 	"../../../include/pifg/common.h"

//int timeOutRetry = 0;

//	External classes ________________________________________________________
extern CLog	oLog;

#define		DBG			1		// 1 : debug message is available

//	Extarnal functions ______________________________________________________

int	getTime();					// get current time in sec.
extern  pthread_mutex_t     mutex_eif;

//  Local variables _________________________________________________________

static CONF	config[] = {
	{ 1,  5, defImageDensity, 	SR_ID,  1},	// #00: Image Density
	{30,144, defFormHeight, 	SR_FH,  1},	// #01: Form Height
	{60,180, defFormWidth, 		SR_FW,  1},	// #02: Form Width
	{ 1,  8, defSuctionDuty, 	SR_SFD, 1},	// #03: Suction Fan Duty
	{ 0,  3, defTransferCurrent,SR_TRC, 1},	// #04: Transfer Current
	{ 0,  1, defSeparatorCHCntl,SR_SCC, 1},	// #05: Separator CH control
	{ 0,  4, defAIDCMode, 		SR_AIM, 0},	// #06: AIDC Mode
	{ 0,  5, defResolution, 	SR_RES, 1},	// #07: Resolution
	{ 0,254, defFineAdjust, 	SR_VFA, 1},	// #08: Vertical Fine Adjustment
	{ 0,254, defFineAdjust, 	SR_HFA, 1},	// #09: Horizontal Fine Adjustment
	{ 0,  1, 0, 				SR_RVM, 1},	// #10: Reverse Mode
	{ 1,  4, defEnergy4Fusion,	SR_EFF, 1},	// #11: Energy For Fusion
	{-1,-1, -1, -1},						// #xx: end mark
};

//	Added by tmatsu on Jan.5, 2016
const int offset = 0x8000;
//	End of change

//	Global methods __________________________________________________________
/*
	<method>		CEif
	<description>	Constructor 
	<args>			int fd = file descriptor
*/
CEif::CEif(int fd)
{
	LOG("libPIF",LogInfo,"Creating CEif class instance ...");
    nFd = fd; 						// save a file descriptor
	port = 0;						// use channel-A
	nActive = 0;					// EIF inactive

//	initialize mutex
	 pthread_mutex_init(&mutex_eif, NULL);

//	set default of configurations
	setDefaultConfig();

#if	DBG
	nDBG = 1;						// enable debug messages
#else
	nDBG = 0;						// disable debug messages
#endif
}

/*
	<method>		getCTS	
	<description> 	get CTS signal	
					Returns CST signal status in the LSR
	<args>			none
	<return>		int = true(1) : active 
						  false(0): inactive 
*/
int	
CEif::getCTS()
{
	int		i, cts1, cts2;
	int 	ret = false;

	cts1 =  getRegister(R_UART_MSRA) & 0x10;		// read Modem Status reg
	for(i = 0; i < 10; i++){
		usleep(0);
		cts2 =  getRegister(R_UART_MSRA) & 0x10;	// read Modem Status reg
		if(cts1 == cts2){
			if(cts1)
				ret = true;
			break;
		}
		cts1 = cts2;
	}
	return(ret);
}

/*
	<method>		getConfig	
	<description>	receive configuration to prionter  
	<args>			none
	<return>		none
*/
void 
CEif::getConfig()
{
	int		i = 0;
	CONF*	ptr = config;

//	receive printer configurations from printer
	while(ptr[i].min >= 0){
		putRequest(ptr[i].cmd,i,0);	
		i++;
	}
}

/*
	<method>		getPageEnd
	<description>	Get number of PAGEEND status
					Returns number of PAGEND status
	<args>			none
	<return>		int = number of PAGEND status 
*/
int	
CEif::getPageEnd()
{
	return(ioctl(nFd, PIF_RD_PAGEND));
}

/*
	<method>		getTotalPageEnd
	<description>	Get total number of PAGEEND status
					Returns total number of PAGEND status
	<args>			none
	<return>		int = number of PAGEND status 
*/
int	
CEif::getTotalPageEnd()
{
	return(ioctl(nFd, PIF_RD_INTC, 11));
}

int
CEif::getWarning()
{
	int	val;

	val = (putRequest(R_PCS,-1,0) & 0x3f) << 9;
	val |= putRequest(R_WRN,-1,0) & 0x1ff;
	return val;
}

/*
	<method>		putOperation
	<description>	Sends a operation command to EIF
	<args>			int	cmd = command
					int par = parameter	
	<return>		int = 	EIF_OK(0)
							EIF_TX_ERROR(-1)		
							EIF_RX_TIMEOUT(-2)
*/
int	
CEif::putOperation(int cmd, int par, char *s)
{
	char	buf[8];
	int		ret;
	ostringstream os;
	char	dst[256];
	int retry;

	if(!par)								// OFF command
		sprintf(buf,"\002F%02d\003",cmd);	
	else
		sprintf(buf,"\002N%02d\003",cmd);	// ON command
	for(int retry = 0; retry < EIF_MAX_RETRY; retry++){
		if((ret = putCommand(buf, EIF_TIMEOUT1,dst)) >= 0 )
			break;
		else if(nActive == false)
			break;
		os.str("");
		os << "EIF Operation command Retry = " << retry;
		LOG("libPIF",LogDebug,os.str());
	}
	return(ret);
}

/*
	<method>		putSetup
	<description>	Tx setup command to EIF
	<args>			int	cmd = command
					int par = parameter	
					int cnf = configuration #
	<return>		int EIF_OK(0)
						EIF_TX_ERROR(-1)		
						EIF_RX_TIMEOUT(-2)
*/
int	
CEif::putSetup(int cmd, int par, int cnf, char *s)
{
	int		ret;
	char	buf[16];
	ostringstream	os;
	char	dst[256];
	int retry;
	
	sprintf(buf,"\002S%02d%X\003",cmd,par);	
	for(int retry = 0; retry < EIF_MAX_RETRY; retry++){
		if((ret = putCommand(buf, EIF_TIMEOUT2,dst)) >= 0 )
			break;
		else if(nActive == false)
			break;
		os.str("");
		os << "EIF Setup command Retry = " << retry;
		LOG("libPIF",LogDebug,os.str());
	}
	if(cnf >= 0)
		nPrinterConf[cnf] = par;
	return(ret);
}


/*
	<method>		putDMC
	<description>	Tx setup command to EIF (Direct MC)
	<args>			int	cmd = command
					int par = parameter	
	<return>		int EIF_OK(0)
						EIF_TX_ERROR(-1)		
						EIF_RX_TIMEOUT(-2)
*/
int	
CEif::putDMC(int cmd, int par, char *s)
{
	int		ret;
	char	buf[16];
	ostringstream	os;
	char	dst[256];
	int retry;
	
	sprintf(buf,"\002S%02d%02d%X\003",S_DMC,cmd,par);	
	for(retry = 0; retry < EIF_MAX_RETRY; retry++){
		if((ret = putCommand(buf, EIF_TIMEOUT2,dst)) >= 0 )
			break;
		else if(nActive == false)
			break;
		os.str("");
		os << "EIF Setup DMC command Retry = " << retry;
		LOG("libPIF",LogDebug,os.str());
	}
	return(ret);
}

/*
	<method>		putRequest
	<description>	Tx status request command to EIF
	<args>			int	cmd = command
					int cnf = configuration #
					int flag = 0 : RX from printer
							   1 : return value in the work area
	<return>		int = RX value (flag = 0)
						  value in the work area (flag = 1)	
*/
int	
CEif::putRequest(int cmd, int cnf, int flag, char *s)
{
	int		ret;
	char	buf[8];
	ostringstream	os;
	char	dst[256];
	int retry;

	if(!flag){
		sprintf(buf,"\002R%02d\003",cmd);	
		for(retry = 0; retry < EIF_MAX_RETRY; retry++){
			if((ret = putCommand(buf, EIF_TIMEOUT1,dst)) >= 0 )
				break;
			else if(nActive == false){
				break;
			}
			os.str("");
			os << "EIF Request cmd(" << hex << cmd << ") Retry = " << dec << retry << "(" << ret << ")";
			LOG("libPIF",LogErr,os.str());
		}
		if((cnf >= 0) && (ret >= 0))
			nPrinterConf[cnf] = ret;
	}
	else{
		ret = nPrinterConf[cnf];
	}
	return(ret);
}

/*
	<method>		resetPagheEnd
	<description>	Reset PAGEEND interrupt counter
	<args>			none
	<return>		none
*/
void 
CEif::resetPageEnd()
{
	ioctl(nFd, PIF_RST_PAGEND);
}

/*
	<method>		setConfig	
	<description>	transmit configuration to prionter  
	<args>			none
	<return>		none
*/
void 
CEif::setConfig()
{
	int		i = 0;
	CONF*	ptr = config;

//	set dafault for printer configurations
	while(ptr[i].min >= 0){
		if(ptr[i].flag)
			putSetup(ptr[i].cmd, nPrinterConf[i],-1);	
		i++;
	}
}

//	Private mathods _________________________________________________________
/*
	<method>		putCommand
	<description>	sends a command to EIF
	<args>			char* buf = pointer to buffer
					int timeout = time out (sec.)
	<return>		int =	EIF_OK(0)
							EIF_TX_ERROR(-1)		
							EIF_RX_TIMEOUT(-2)
*/
int	
CEif::putCommand(char* buf, int timeout, char *s)
{
	int		ret = true;
	char	rxbuf[16],*ptr;
	int		start;
	int		c,n;
	char	cmd;
	char	dst[256];
	int retry;

	if(nActive == false)
		return(EIF_RX_TIMEOUT);

//	Tx command
	pthread_mutex_lock(&mutex_eif);
	if(nActive){
		if((ret = puts(buf,offset,dst)) != true){	
			LOG("libPIF",LogErr,"EIF Tx Error !");
			pthread_mutex_unlock(&mutex_eif);
			return(EIF_TX_ERROR);					// return if failed
		}
//	Wait for answer
		start = getTime();							// start timer
		while(true){
			if((c = getchar(offset,dst)) == C_STX){ //start of text
				ptr = rxbuf;
				cmd = getchar(offset,dst);					// RX command 
				for(n = 0; n < 15; n++){			// RX parameter
					if((c = getchar(offset,dst)) == C_ETX){ //end of text
						*ptr = '\0';
						break;
					}
					*ptr++ = toupper(c);
				}	
				break;
			}
			else if(abs(getTime() - start) > timeout){
				LOG("libPIF",LogErr,"EIF Rx Time Out !");
				cout << "EIF Rx Time Out:" << buf << endl;
				pthread_mutex_unlock(&mutex_eif);
				nActive = false;	
				return(EIF_RX_TIMEOUT);				// return if error
			}
		}

//	Analyze RX packet
		if(cmd == ST_ANSWER)						// RX answer
			ret = rxbuf[0] - '0';	
		else if(cmd == ST_STATUS){					// RX status
			for(ret = 0,c = 0; c < n; c++) {
				ret <<= 4;							// *= 16
				if(rxbuf[c] > '9') 	
					ret += (rxbuf[c] - '7') ;
				else 				
					ret += (rxbuf[c] - '0');
			}
		}
		else										// illegal status
			ret = EIF_RX_ILLEGAL;
	}
	else
		ret = EIF_RX_TIMEOUT;
	if(ret < -3){	
		ostringstream os;
		os << "putCommnd: " << rxbuf << " (" << n << ")";
		LOG("libPIF",LogInfo,os.str().c_str());
	}		
	
//cout << "\tRX: " << ret << endl << flush;
	pthread_mutex_unlock(&mutex_eif);
	return (ret);
}

/*
	<method>		setDefaultConfig
	<description>	Set default value of configurations
	<args>			none
	<return>		none
*/
void 
CEif::setDefaultConfig()
{
	int		i = 0;
	CONF*	ptr = config;

//	set dafault for printer configurations

	while(ptr[i].min >= 0){
		nPrinterConf[i] = ptr[i].def;		// set default	
		i++;
	}
}

