///////////////////////////////////////////////////////////////////////////////
//	Module			: CIsd202.C
//	Description		: Implementation of CIsd202 class. 
//                    This class is a delived from CSerial class and
//                    provides functionality to communicate with Operation 
//					  panel.
//
//	Compiler		: gcc 4.4.7++
//	Author			: T.Matsuzawa of JEI Matsumoto Lab.
//	E-mail			: tmatsu.jei@nifty.com
//- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
// Maintenance:
//	Jan.15, 2014	Renewal for CentOS6.5/64bit
///////////////////////////////////////////////////////////////////////////////	
#include 	<iostream>
#include 	<fstream>
#include 	<sstream>
using namespace std;
#include	<unistd.h>
#include	<stdlib.h>

#include 	"../../include/pifg/libPIF.h"
#include <pthread.h>
#include <stdio.h>
//#include 	"pifg/common.h"
//#include 	"pif/CIsd202.h"
#include 	"../../include/pifg/CLog.h"
#include <../../home/Genesis/include/lis.h>


//	External classes ________________________________________________________
extern CLog	oLog;
int MBVersion;     // checking mother board version, Jetway(New) = 2, Gigabyte(Old) = 1

extern int printdataOn;

//	External functions _____________________________________________________

int	getTime();					// get current time in sec.

//  Local variables _________________________________________________________

const	int		Beep1[] = {30,30,30,0};
const	int		Beep2[] = {5,5,5,5,5,5,5,100,5,5,5,5,5,5,5,0};
const	int		Beep3[] = {5,5,5,5,5,0};
const	int*	Beep[] = { Beep1, Beep2, Beep3};

//		commands
const	char	write_start[] = "Ud";			// start of write mode
const	char	write_end[] = "Uc\r";			// end of write mode
const	char	page_start[] = "FS\r";			// start of page transfer 
const	char	page_end[] = "Fs\r";			// end of page transfer 
const	char	page_erase[] = "Fpageerase\r";	// page erase 
const	char	bmp_erase[] = "Fbiterase\r";	// bitmap data erase 
const	char	font_start[] = "FG\r";			// start of page transfer 
const	char	font_end[] = "Fg\r";			// start of page transfer 
const	char	font_erase[] = "Fgaijierase\r";	// font data erase 

const	char	get_csp[] = "Fb\r";				// get checksum of page data  
const	char	get_csf[] = "Fd\r";				// get checksum of flash rom  

const	char	ans_page_start[] = "Data Save Start";
const	char	ans_page_end[] = "Data Save End";
const	char	ans_page_erase[] = "PageData Erase Start";
const	char	ans_page_erase_end[] = "PageData Erase End";
const	char	ans_bmp_erase[] = "BitmapData Erase Start";
const	char	ans_bmp_erase_end[] = "BitmapData Erase End";
const	char	ans_font_start[] = "Gaiji Save Start";
const	char	ans_font_end[] = "Gaiji Save End";
const	char	ans_font_erase[] = "Gaiji Erase Start";
const	char	ans_font_erase_end[] = "Gaiji Erase End";

const	char	ans_mdw_start[] = "MDW_START";
const	char	ans_mdw_end[] = "MDW_END";

int     updateFlashFlag;

//	Global methods _________________________________________________________
/*
	<method> 		CIsd202()
	<description>	Constructor 
	<arguments>		int fd = file descriptor
*/
CIsd202::CIsd202(int fd)
{
    string lineFromFile;
	LOG("libPIF",LogInfo,"Creating CIsd202 class instance ...");
     	ifstream mbVersion("/home/Genesis/motherBoardVersion.txt");
    	if(mbVersion.is_open()){
        	getline(mbVersion,lineFromFile); 
        MBVersion = atoi(lineFromFile.c_str());
    	}
    nFd = fd;							// save a file descriptor
	port = 1;							// use channel-B

	pthread_mutex_init(&_ISDWaiterMutex,NULL);
	pthread_cond_init(&_ISDWaiterCond,NULL);
	
	putBaud(BAUD_38400);				// set baud rate 38400 bps
	putTrigger(TRIGGER_8);				// set RX trigger level to 8bytes 
    if(!waitReady()){					// wait until panel gets ready
		nConnect = false;
	}
	else{
		initialize();
	}
}
//		General methods
/*
	Function 	: KeyIntrOn()
	Description	: minus a key intr count
	Arguments	: None
	Returns		: None
*/
void CIsd202::KeyIntrOn()
{
	if(_keyIntrCnt)
	{
		_keyIntrCnt--;
		if(!_keyIntrCnt)
		{
			_keyIntrOn=1;
		}
	} else {
		printf("<<< OVER KeyIntrOn()!!! >>>\n");
	}
}

/*
	Function 	: KeyIntrOff()
	Description	: adds a key intr count
	Arguments	: None
	Returns		: None
*/
void CIsd202::KeyIntrOff()
{
	_keyIntrOn=0;
	_keyIntrCnt++;
}

/*
	Function 	: disable()
	Description	: Disable panel
	Arguments	: None
	Returns		: None
*/
void 	CIsd202::disable()
{
	txCommand("STOP");               	// stop panel
}

/*
	Function 	: enable()
	Description	: Enable panel
	Arguments	: None
	Returns		: None
*/
void 	CIsd202::enable()
{
	txCommand("RUN");               	// restart panel
}


//		Diplay-related methods
/*
	<method> 		clearScreen()
	<description>	Clear all screen
	<arguments>		None
	<return>		None
*/

/*void	CIsd202::clearScreen()
{
	char	buf[32];

	sprintf(buf,"DE,%X",nBG);
	txCommand(buf);
}*/
void	CIsd202::clearScreen()///
{
	char	buf[32];
	static int already = 0;

	if( already ) {
		clearPage();
	} else {
		sprintf(buf,"DE,%X",nBG);
		txCommand(buf);
	}
}

void	CIsd202::clearPage()
{
	char	buf[32];

	if( _lastPage >= 3 && _lastPage <= 9 ) {
		dispPage( _lastPage + 10 );
		dispPage( _lastPage - 10);
	} else if(_lastPage == 20) {	// New Menu w/ Select All
		dispPage( _lastPage + 10 );
		dispPage( _lastPage - 10);
	} else {
		sprintf(buf,"DE,%X",nBG);
		txCommand(buf);
	}
}

/*
	Method 		: display() - 1
	Description	: Display a character at specified position.
	Arguments	: int x = x address (column)
				  int y = y address (line)
				  char c = character
	Returns		: None
*/
void	CIsd202::display(int x, int y, char c)
{
	char	buf[32];

	sprintf(buf,"DO,%d,0,%d,%d,%X,%X,'%c",nSize,x,y,nFG,nBG,c);
	txCommand(buf);
}

/*
	Method 		: display() - 2
	Description	: Display string at specified position.
				  String must be terminated by NULL.
	Arguments	: int x = x address (column)
				  int y = y address (line)
				  const string str = string
	Returns		: None
*/

void	CIsd202::display(int x, int y, const string str)
{
	//ostringstream oss;
	char buf[1024];

	//oss << "DO," << nSize << ",0," << x << "," << y << ",";
	//oss << hex << uppercase << nFG << "," << nBG << ",'" << str << ends << dec; 
	//txCommand(oss.str());
	if(*str.c_str())
	{
		sprintf(buf,"DO,%d,0,%d,%d,%X,%X,'%s",nSize,x,y,nFG,nBG,str.c_str());
		txCommand(buf);
	}
}

void	CIsd202::display(const POINT& p, const string str)
{
	//ostringstream oss;
	char buf[1024];

	//oss << "DO," << nSize << ",0," << p.x << "," << p.y << ",";
	//oss << hex << uppercase << nFG << "," << nBG << ",'" << str << ends << dec; 
	//txCommand(oss.str());
	if(*str.c_str())
	{
		sprintf(buf,"DO,%d,0,%d,%d,%X,%X,'%s",nSize,p.x,p.y,nFG,nBG,str.c_str());
		txCommand(buf);
	}
}

void	CIsd202::display(int x, int y, char *str)
{
	char	buf[1024];

	if(str)
	{
		if(*str)
		{
			sprintf(buf,"DO,0,0,%d,%d,%X,%X,'%s",x,y,nFG,nBG,str);
			txCommand(buf);
		}
	}
}

void	CIsd202::displayRev(int x, int y, char *str, int rev)
{
	char	buf[1024];

	if(str)
	{
		if(*str)
		{
			if( rev ) sprintf(buf,"DO,0,0,%d,%d,%X,%X,'%s",x,y,nBG,nFG,str);
			else sprintf(buf,"DO,0,0,%d,%d,%X,%X,'%s",x,y,nFG,nBG,str);
			txCommand(buf);
		}
	}
}

void	CIsd202::displayRevCustomColor(int x, int y, char *str, int rev, int fgColor, int bgColor)
{
	char	buf[1024];

	if(str)
	{
		if(*str)
		{
			if( rev ) sprintf(buf,"DO,0,0,%d,%d,%X,%X,'%s",x,y,fgColor,bgColor,str);
			else sprintf(buf,"DO,0,0,%d,%d,%X,%X,'%s",x,y,fgColor,bgColor,str);
			txCommand(buf);
		}
	}
}

void	CIsd202::displays(int x, int y, int size, int spacing, char *str)
{
	char	buf[1024];

	if(str)
	{
		if(*str)
		{
			sprintf(buf,"DO,%d,%d,%d,%d,%X,%X,'%s",size,spacing,x,y,nFG,nBG,str);
			txCommand(buf);
		}
	}
}

/*
	Function 	: dispPage()
	Description	: Disply a page
	Arguments	: int page = page#
	Returns		: None
*/
void 	CIsd202::dispPage(int page)
{
	char	buf[16];
	
	_lastPage = page;

    sprintf(buf,"MWS,Page,%d",page);
    txCommand(buf);
}

/*
	Method 		: drawBox()
	Description	: Draw box
	Arguments	: POS& pos = (x1,y1),(x2,y2)
				  int c = color
	Returns		: None
*/
void	CIsd202::drawBox(POS& pos, int c)
{
	char	buf[32];

	sprintf(buf,"DB,%d,%d,%d,%d,%X,%X",pos.x1,pos.y1,pos.x2,pos.y2,c,c);
	txCommand(buf);
}

void CIsd202::drawBox(int x1, int y1, int x2, int y2, int c)
{
	char	buf[32];

	sprintf( buf, "DB,%d,%d,%d,%d,%X,%X", x1, y1, x2, y2, c, c );
	txCommand(buf);

	return;
}

void CIsd202::drawCustomLine(int x1, int y1, int x2, int y2, int frameColor)
{
	char	buf[32];

	sprintf( buf, "DL,%d,%d,%d,%d,%X", x1, y1, x2, y2, frameColor );
	txCommand(buf);

	return;
}

void CIsd202::drawDot(int x, int y, int c)
{
	char	buf[32];

	sprintf( buf, "DD,%d,%d,%X", x, y, c );
	txCommand(buf);
}

void CIsd202::drawLine(int x1, int y1, int x2, int y2, int c)
{
	char	buf[32];

	//if( x1 > x2 ) { x1 ^= x2; x2 ^= x1; x1 ^= x2; }
	//if( y1 > y2 ) { y1 ^= y2; y2 ^= y1; y1 ^= y2; }
	sprintf( buf, "DL,%d,%d,%d,%d,%X", x1, y1, x2, y2, c );
	txCommand(buf);
}

void CIsd202::drawRect(int x1, int y1, int x2, int y2, int c)
{
	drawLine( x1, y1, x2, y1, c );
	drawLine( x1, y1, x1, y2, c );
	drawLine( x2, y1, x2, y2, c );
	drawLine( x1, y2, x2, y2, c );
}

void CIsd202::drawPicture(int x, int y, int ndx)
{
	char	buf[32];

	sprintf( buf, "DI,%d,%d,%d", x, y, ndx );
	txCommand(buf);
}

void CIsd202::drawEclipse(int cx, int cy, int rx, int ry, int fill_col, int frame_col)
{
	char	buf[32];

	sprintf( buf, "DC,%d,%d,%d,%d,%d,%d", cx, cy, rx, ry, fill_col, frame_col );
	txCommand(buf);
}

void CIsd202::drawEclipse2(int x1, int y1, int x2, int y2, int fill_col, int frame_col)
{
	char	buf[32];

	sprintf( buf, "DC,%d,%d,%d,%d,%d,%d", (x1+x2) >> 1, (y1+y2) >> 1, abs(x2-x1) >> 1, abs(y2-y1) >> 1, fill_col, frame_col );
	txCommand(buf);
}

void CIsd202::drawCircle(int cx, int cy, int r, int fill_col, int frame_col)
{
	char	buf[32];

	sprintf( buf, "DC,%d,%d,%d,%d,%d,%d", cx, cy, r, r, fill_col, frame_col );
	txCommand(buf);
}
/*
	Method	 	: setColor()
	Description	: Set Color for testr
	Procedure	:
	Arguments	: int fg = foreground color
				  int bg = background color
	Returns		: None
*/
void	CIsd202::setColor(int fg, int bg)
{
	nFG = fg;
	nBG = bg;
}

void CIsd202::myprint()
{
	printf("My:myprint:::type=%d, cmd=%d, par=%d\n",_st.type,_st.cmd,_st.par);
}

void CIsd202::DrawClock(int xx, int yy, int rad, int rev, int rev2, int col, int col2)
{
	int x, y;
	static int cc=0;
	
	//drawCircle( xx, yy, 10, nBG, nFG );
	//drawCircle( xx, yy, 10, 0, nFG );
   	switch(rev)
   	{
   		case 0:	 x = 0;	 y = -5; break;
   		case 1:	 x = 3;	 y = -4; break;
   		case 2:	 x = 4;	 y = -3; break;
   		case 3:	 x = 5;	 y = 0; break;
   		case 4:	 x = 4;	 y = 3; break;
   		case 5:	 x = 3;	 y = 4; break;
   		case 6:	 x = 0;	 y = 5; break;
   		case 7:	 x = -3; y = 4; break;
   		case 8:	 x = -4; y = 3; break;
   		case 9:	 x = -5; y = 0; break;
   		case 10: x = -4; y = -3; break;
   		case 11: x = -3; y = -4; break;
   	}
   	if( (cc & 1) == 0 ) {
	   	if( ((cc/24)%2) == 1 )
			drawLine( xx, yy, xx + (x * 2), yy + (y * 2), col );
		else
			drawLine( xx, yy, xx + (x * 2), yy + (y * 2), /*nBG*/col2 );
	} else {
		drawLine( xx, yy, xx + (x * 2), yy + (y * 2), nBG );
	}
	cc++;
   	switch(rev2)
   	{
   		case 0:	 x = 0;	 y = -5; break;
   		case 1:	 x = 3;	 y = -4; break;
   		case 2:	 x = 4;	 y = -3; break;
   		case 3:	 x = 5;	 y = 0; break;
   		case 4:	 x = 4;	 y = 3; break;
   		case 5:	 x = 3;	 y = 4; break;
   		case 6:	 x = 0;	 y = 5; break;
   		case 7:	 x = -3; y = 4; break;
   		case 8:	 x = -4; y = 3; break;
   		case 9:	 x = -5; y = 0; break;
   		case 10: x = -4; y = -3; break;
   		case 11: x = -3; y = -4; break;
   	}
	//drawLine( xx, yy, xx + (x * 2), yy + (y * 2), col2 );

	if( ++rev >= 12 ) {
		rev = 0;
		if( ++rev2 >= 12 ) {
			rev2 = 0;;
		}
	}
	
}

void CIsd202::drawColorMatrix()
{
	for(int b=0; b < 8; b++) {
   	for(int a=0; a < 8; a++) {
		drawBox( 20*a, 		15*b, 		20*a+19, 		15*b+14, 		((7-b) << 5)+((7-a) << 2) + 0);
		drawBox( 160 + 20*a,15*b, 		160 + 20*a+19,	15*b+14, 		((7-b) << 5)+(a << 2) + 1);
		drawBox( 20*a, 		120 + 15*b, 20*a+19, 		120 + 15*b+14,	(b << 5)+((7-a) << 2) + 3);
		drawBox( 160 + 20*a,120 + 15*b, 160 + 20*a+19,	120 + 15*b+14,	(b << 5)+(a << 2) + 2);
		///drawBox( 20*a, 16*b, 20*a+19, 16*b+15, (((b*16+a) & 7) << 5) + ((((b*16+a) >> 4)  & 7) << 2)+((b>>2) & 2) + ((a>>3) & 1));
   	}
    }
}

//	Key input

/*
	Function 	: getKey()
	Description	: Wait for key
	Procedure	:
	Arguments	: ISDST& st = reference to ISDST
	Returns		: int true = accepted
					  false = error
*/
int	CIsd202::getKey_FWFlash(ISDST& st)
{
    while(isEmpty()){}
	return(eatPacket(st));
}

int	CIsd202::getKey(ISDST& st)
{
	int result;
	int rate = 20;
		
    	while(1)
   	{
		if(_keyLocked)
		{
			
		} else 
		{
		if(!isEmpty((char*)"getKey:")) {
                	if (updateFlashFlag == 1)
                    	    printf("getKey: Break;");
                	else
                    	    break;
                }
		}
		if (MBVersion == 2)   //Jetway(New) Motherboard
            		usleep(1000000/rate);
        else if ((MBVersion == 1) && (printdataOn != 0)) {  //only to sleep on printing, if not pritning this sleep time make TDU menu to response slow, version 60021
                usleep (10000);
        }
    }
    pthread_cond_signal(&_ISDWaiterCond);
    _keyPresses++;
    result = eatPacket(st,(char*)"gtKy");
	if(st.type) {
		_st.type = st.type;
		_st.cmd = st.cmd;
		_st.par = st.par;
	}
    
	return result;
}

void CIsd202::waitKey()
{
	pthread_mutex_lock(&_ISDWaiterMutex);
	pthread_cond_wait(&_ISDWaiterCond,&_ISDWaiterMutex);
	pthread_mutex_unlock(&_ISDWaiterMutex);
}

/*int	CIsd202::getKey2(ISDST *st)
{
	int result;
	static int	rev=0, rev2=0;
	static int xx = 0, yy = 0;
	static int dc = 0;
	int rate = 20;
	
    while(isEmpty())
    {
    	///*if( dc++ < (rate*10/2) );	//Do Nothing
    	//else {
    	
*/
/*    		usleep(1);
    		dc = 0;
			DrawClock( 245 + xx*20, 10 + yy*20, 10, rev/2, rev2/2, 3, 224 );
			if( ++rev >= 24 ) {
				rev = 0;
				if( ++rev2 >= 24 ) {
					rev2 = 0;
				}
			}*/
			//drawDot(0,0,0);
		//}*/
/*		usleep(1000000/rate);
    }
    result = eatPacket(*st);
	if(st->type) {
		_st.type = st->type;
		_st.cmd = st->cmd;
		_st.par = st->par;
	}
    
	return result;
}
*/
int	CIsd202::getLastKey(ISDST& st) {			// get last key code
	if(st.type) {
		st.type = _st.type;
		st.cmd = _st.cmd;
		st.par = _st.par;
	}
	if( _st.type == 'A' ) {
		///_st.type = _st.cmd = _st.par = 0;
		return 1;
	} else {
		return 0;
	}
}
int CIsd202::isKeyPressed() {
	return _st.type;
}

/*int CIsd202::experiment(ISDST *st) {
//int CIsd202::experiment() {
	return 0;
}*/
int CIsd202::getKeyPressed() {
	int key;
	if( _st.type == 'A' ) {
		key = _st.cmd;
		//printf("key=%d, %x\n", key, key );
		_st.type = _st.cmd = _st.par = 0;
	/*	switch(key) {
			
			default:
				key = key;
		}*/
		return key;
	} else {
		return 0;
	}
}

int CIsd202::waitKeyPresses()
{
	int rate = 20;
	while(!_keyPresses && !_redisplayMenu)
	{
	if(MBVersion == 2)
            usleep(1000000/rate);
    	else if(MBVersion == 1)
            usleep(1000);
   //         usleep(10000);
	}
	if ((MBVersion ==1) && (_keyPresses))
        usleep(10000);   // added for menu delay response in old motherboard,version 60021
     //       usleep(15000);  // this is also a good value to test 
    
	_keyPresses=0;
}

/*
	Function 	: WaitKey()
	Description	: Wait for key
	Arguments	: int key = key code  (0: any keys)
				  int flag = 0 : wait forever
				  			 1 : return immediately
	Returns		: int true = specified key is pressed
*/
int	CIsd202::waitKey(int key, int flag)
{
	ISDST	st;

	while(true) {
		if(isEmpty((char*)"waitKey:") == false){
			if(eatPacket(st,(char*)"wtKy")){
				if((key == st.cmd) || !key)
					break;
				else if(st.cmd == 'R')
					return false;
			}
			else
				return false;
		}
		else if(flag == 1)
			return false;
	}
	return true;
}

//	Sound-related methods
/*
	Function 	: beeper()
	Description	: beep controller
	Arguments	: int pattern = pattern#
	Returns		: None
*/
void 	CIsd202::beeper(const int pattern)
{
	int	i,j;
	int const*	ptr = Beep[pattern];

	j = 1;
	while((i = *ptr++) != 0){
		if(j & 1)
			buzzer(i);
		usleep(i * 1000);
		j++;
	}
}

/*
	<method> 		buzzer()
	<description>	Buzzer control  
	<arguments>		int ms = active time in ms
	<return>		None
*/
void	CIsd202::buzzer(const int ms)
{
	char	buf[8];

	if(ms > 0){
		sprintf(buf,"DR,%d",ms);   
		txCommand(buf);
	}
}

/*
	<method> 		buzzer()
	<description>	Interval buzzer control  
	<arguments>		int on = active time in ms
					int off = inactive time in ms
	<return>		None
*/
void	CIsd202::buzzer(const int on, const int off)
{
	char	buf[16];

	sprintf(buf,"dr2,%d,%d",on,off);   
	txCommand(buf);
}

//	Memory access

/*
	Method 		: readMemory()
	Description	: Read a value in the panel memory
	Procedure	:
	Arguments	: int idx = index of memory 
	Returns		: int val = value
*/
int		CIsd202::readMemory(int idx)
{
	char	buf[16];
	ISDST	st;
	int		val = -1;

	sprintf(buf,"MR,MS%d",idx);
	txCommand(buf);                					// send read command
	if(eatPacket(st,(char*)"rdMem") == true){     					// Rx reply
		if(st.type == 'R')
			val = st.par;
	}
	return val;
}

/*
	Method 		: writeMemory()
	Description	: Write value in the panel memory
	Arguments	: int idx = index of memory 
				  int val = value
	Returns		: None
*/
void	CIsd202::writeMemory(int idx, int val)
{
	char	buf[16];

	sprintf(buf,"MWS,MS%d,%d",idx,val);
	txCommand(buf);
}

/*
    Method      : writeMemory()
    Description : Write text string in the panel memory
    Arguments   : int idx = index of memory
                  char* ptr = pointer to text string
    Return      : None
*/
void    CIsd202::writeMemory(int idx, char* ptr)
{
    char    buf[64];

    sprintf(buf,"MWT,MT%d,'%s",idx,ptr);
    txCommand(buf);
}    

/*
	Method 		: getChecksum()
	Description	: Get checksum of ISD202
	Arguments	: int ctrl = 0: all area 
							 1: imagearea 
	Returns		: None
*/
int	CIsd202::getChecksum(int ctrl)
{
	char	buf[16];
	int	checksum = 0;
    //updateFlashFlag = 1;
	LOG("libPIF",LogInfo,"Start write mode");
	txCommand(write_start);		// start write mode
	sleep(3);					// wait for reboot (3 sec.)

	if(!ctrl)
		puts(get_csf,0,(char*)"gtChS1");			// get checksum of whole flash rom
	else if(ctrl == 1)
		puts(get_csp,0,(char*)"gtChS2");			// get checksum of image area 
	else
		return checksum;
	//if(rxAnswer(buf,5)){
    rxAnswer(buf,15);
    cout << "> checksum = " << buf << endl;
    buf[2] = '\0';
    sscanf(buf,"%x",&checksum);	
    //}
	LOG("libPIF",LogInfo,"End write mode");
	puts(write_end,0,(char*)"gtChS3");			// end write mode
    //updateFlashFlag = 0;
	sleep(3);
	return	checksum;
}


/*
	Method 		: updateFlash()
	Description	: Update flash memory on the ISD-202
	Arguments	: char* fname = pointer to filename
				  int flag = 1: graphics data exists
	Returns		: None
*/
int	CIsd202::updateFlash(char* fname, int flag)
{
	char	buf[80];
	char	lbuf[120];
	
    updateFlashFlag = 1;
	sprintf(buf,"Open page data file: %s",fname);
	LOG("libPIF",LogInfo,buf);
	ifstream	fin(fname);
	if(!fin){
		sprintf(buf,"Cannot open a file: %s",fname);
		LOG("libPIF",LogErr,buf);
		return false;
	}
    	printf("\twrite_start\n");
	LOG("libPIF",LogInfo,"Start write mode");
	txCommand(write_start);		// start write mode
	sleep(3);					// wait for reboot (3 sec.)
    	printf("\tpage_start\n");
	puts(page_start,0);			// start page transfer mode
	if(rxAnswer(buf,5)){
        printf("\tpage_start Answer = %s \n", buf);
		if(strcmp(buf,ans_page_start))
            sleep(1);
			//return false;
		else{
			sprintf(lbuf,"> %s",buf);
			LOG("libPIF",LogInfo,lbuf);
		}
	}
//	else
//		return false;
//    sprintf(buf,"Start write mode");
    printf("\tpage_erase Start\n");
	puts(page_erase,0);				//  page erasing ...  
	if(rxAnswer(buf,5)){
        printf("\tpage_erase Start Answer = %s \n", buf);
		if(strcmp(buf,ans_page_erase)){
            //return false;
        }else{
			sprintf(lbuf,"> %s",buf);
			LOG("libPIF",LogInfo,lbuf);
		}
	}
//	else
//		return false;

    printf("\tpage_erase End\n");
	if(rxAnswer(buf,30)){
        printf("\tpage_erase end Answer = %s \n", buf);
		if(strcmp(buf,ans_page_erase_end))
			return false;
		else{
			sprintf(lbuf,"> %s",buf);
			LOG("libPIF",LogInfo, lbuf);
		}
	}

//	else
//		return false;

	if(flag){
        printf("\tbmp_erase\n");
		puts(bmp_erase,0);				//  bitmap erasing ... 
		if(rxAnswer(buf,10)){
            printf("\tbmp_erase Start Answer = %s \n", buf);
			if(strcmp(buf,ans_bmp_erase)){
                return false;
            }else{
				sprintf(lbuf,"> %s",buf);
				LOG("libPIF",LogInfo, lbuf);
			}
		}
		else
			return false;
		if(rxAnswer(buf,30)){
            printf("\tbmp_erase end Answer = %s \n", buf);
			if(strcmp(buf,ans_bmp_erase_end))
				return false;
			else{
				sprintf(lbuf,"> %s",buf);
				LOG("libPIF",LogInfo,lbuf);
			}
		}
        
//		else
//			return false;
	}

	while(!fin.eof()){
		fin.getline(buf,77,C_LF);
		strcat(buf,"\n");
		puts(buf,0);	
		if((buf[0] == 'B') || (buf[0] == 'P')){
			if(rxAnswer(buf,1)){
				sprintf(lbuf,"> %s",buf);
				LOG("libPIF",LogInfo, lbuf);
			}
		}
	}
	fin.close();

	puts(page_end,0);				// end page transfer mode
	if(rxAnswer(buf,5)){	
		if(strcmp(buf,ans_page_end))
			return false;
		else{
			sprintf(lbuf,"> %s",buf);
			LOG("libPIF",LogInfo,lbuf);
		}
	}
//	else
//		return false;
	LOG("libPIF",LogInfo,"End write mode");
	puts(write_end,0);			// end write mode
	sleep(3);
    //system("touch /home/Genesis/flags/FlashPanelUpdateVer.1.0.2");
	updateFlashFlag = 1;
	return	true;
}

void
CIsd202::restart()
{
	initialize();
}

// Private methods ________________________________________________________

void
CIsd202::initialize()
{
	nConnect = true;
	nBG = 0 + (0 <<3) + (0 << 6) + 4 + 32;					// default foregroung text color
	clearScreen();					// clear screen

	///nFG = BLACK;					// default foregroung text color
	///nBG = WHITE;                   	// default background text color
	nFG = ISD_BLACK;					// default foregroung text color
	nBG = ISD_WHITE;                   	// default background text color
	
	nSize = S_NORMAL;				// default character size
	_keyLocked=0;
	_keyIntrOn=1;
	_keyIntrCnt=0;
	_keyPresses=0;
	_redisplayMenu=0;
	_menuInside = 0;
}

/*void	CIsd202::calChecksum(char* src, char* dest)
{
}*/
void	CIsd202::calChecksum(const char* src, char* dest)
{
	int		checksum = 0;
	int		c;
	
	while((c = (int)*src) != '\0'){
		checksum ^= c & 0xff;
		src++;
	}  
	sprintf(dest,"%02x",checksum);	   
}

void CIsd202::ConsumeExcess()
{
	//pthread_mutex_lock( &_mutex );
	while( !isEmpty((char*)"ConsumeExcess:") ) {
		getchar(0,(char*)"ConEx");								// Rx trailing characters
///???		printf( "<<->>" );
	}
	//pthread_mutex_unlock( &_mutex );
}

/*
	Method 		: eatPacket()
	Description	: Analyse command packet
	Arguments	: ISDST &cmd = reference to ISDST structure
	Returns		: int true = normal completion
					  false = Rx Error
*/
int		CIsd202::eatPacket(ISDST &st, char *s)
{
	int		i,j,c;
	int		par;
	char	tmp[32];
	char	csbuf[4];
	char	csbuf2[4];
	int		checksum = 0;
	int		cscount = false;
	char	dst[256];
	char	str[256];
	int		lid;

	///////////////////////////
	concats(dst,s,(char*)"etPk");
	lid=BeginCmdSeq(dst);///
	sprintf(str,"etPk[%d]",lid);
	concats(dst,s,str);
	j = 0;
	i = 0;
	st.type = 0;
	st.cmd = 0;
	st.par = 0;
	////////////////////////////
	while(true) {
		if((c = getchar(0,dst)) == C_STX){				// Hunting STX code ...
			cscount = true;						// Start to calc. checksum
			break;								// Break if STX
		}
		else if(c == false) {
			ConsumeExcess();
			EndCmdSeq(lid,dst);
			return false;						// Return if Rx Error
		} else { // mine
			printf("WOAH! WOAH! WOAH! WOAH! WOAH! WOAH!\n");
			printf("WOAH! WOAH! WOAH! WOAH! WOAH! WOAH!\n");
			printf("WOAH! WOAH! WOAH! WOAH! WOAH! WOAH!\n");
			printf("WOAH! WOAH! WOAH! WOAH! WOAH! WOAH!\n");
			printf("WOAH! WOAH! WOAH! WOAH! WOAH! WOAH!\n");
			printf("WOAH! WOAH! WOAH! WOAH! WOAH! WOAH!\n");
			printf("WOAH! WOAH! WOAH! WOAH! WOAH! WOAH!\n");
			cscount = true;
			tmp[i++] = c;						// store RX data
			checksum ^= c;					// calculate checksum
			break;
		}
	}
	for(i = 0; i < 32; i++) {
		c = getchar(0,dst);							// Rx next character
		///if(c == false)
		///	return false;						// Return if Rx Error
		if(c == false) {
			ConsumeExcess();
			EndCmdSeq(lid,dst);
			return false;						// Return if Rx Error
		}
		if(c == C_LF){
			csbuf[j] = '\0';
			break;
		} 										// break if LF
		if(c == C_ETX){
			cscount=false; 						// Stop to calc. checksum ETX
			j = 0;
			tmp[i] = '\0';
		}
		else if (c >= ' '){
			if(cscount == true){
				tmp[i] = c;						// store RX data
				checksum ^= c;					// calculate checksum
			}
			else if(j < 2)
				csbuf[j++] = c;
		}
	}
	if(c != C_LF) {									// Return if packet broken
		ConsumeExcess();
		EndCmdSeq(lid,dst);
		return false;
	}
	sprintf(csbuf2,"%02x",checksum);
	
	if(strcmp(csbuf,csbuf2)){
		///
		ConsumeExcess();
		EndCmdSeq(lid,dst);
		return false;							// Return if check sumerror
	} else {
	}
	st.type = tmp[0];							// 1st byte = type
	switch(st.type){
		case 'r':	break;
		case 'A':	        					// memory action
			st.cmd = tmp[2];
			st.par = 0;
			i = 3;
			while(tmp[i])
				st.par = st.par * 10 + tmp[i++] - '0';
			break;
		case 'R':								// reply
			st.par = getVal(tmp);
			break;
	}
	ConsumeExcess();
	EndCmdSeq(lid,dst);
	return true;
}

/*
	Function 	: getVal()
	Description	: Get value in the replied string
	Procedure	:
   	Arguments	: char* ptr = pointer to string
	Returns		: int val
*/
int	CIsd202::getVal(char* ptr)
{
	int		val = 0;
	int		i;

	while(*ptr++ != '='){}		// hunt '='
	while(*ptr){
		if((val >= '0') && (val <= '0'))
			val = (val * 10) + (*ptr++ - '0');
		else
			ptr++;
	}
	return val;
}

/*
	Method 		: rxAnswer()
	Description	: Receive answer form ISD-202
	Arguments	: char* buf = pointer to buffer
				  int timeout = time out (sec.)
	Returns		: int true = normal completion
					  false = Time out
*/
int		CIsd202::rxAnswer(char* buf, int timeout)
{
	int	c;
	char*	buf2 = buf;
	int	start = getTime();						// start timer

	while((c = getchar(0,(char *)"rxAn1")) != C_LF){
	if(c){
            cout << (char) c;
	if(c != C_CR)
		*buf++ = c;
	}
	else if(abs(getTime() - start) > timeout){
		cout << "O.P Rx Time Out !" << endl;
            *buf = '\0';
            getchar(0,(char *)"rxAn2");
			return false;				// return if error
		}
	}
	*buf = '\0';
	getchar(0,(char *)"rxAn2");
	return true;
}

/*
	<method> 		txCommand()
	<description>	Send command string
	<arguments>		const char* ptr = pointer to string
	<return>		None
*/
void	CIsd202::txCommand(const char* ptr,char *s)
{
	int		n = strlen(ptr);
	//char	buf[80];
	char	buf[256];
	char	dst[256];
	char	str[256];

	_scratchStatus=0;
	buf[0] = C_STX;	
	// packet header
	_scratchStatus=1;
	strcpy(&buf[1],ptr);					// command string
	_scratchStatus=2;
	buf[n+1] = C_ETX;							// end of command
	_scratchStatus=3;
	calChecksum(ptr,&buf[n+2]);				// checksum
	_scratchStatus=4;
	buf[n+4] = C_CR;							// CR  
	_scratchStatus=5;
	buf[n+5] = '\0';						// NULL terminate
	_scratchStatus=6;
	//concats(dst,s,"txCmd1");
	_scratchStatus=7;
	//printf("<<< txCommand[%d,%d] = %s>>>\n",strlen(buf),strlen(ptr),ptr);
	sprintf(str,"txCmd1[%d,%d] = --<%s>--\n",strlen(buf),strlen(ptr),ptr);
	concats(dst,s,str);
	_scratchStatus=8;
	puts(buf,0,dst);	
	_scratchStatus=9;
}
//TEMPO
/*void	CIsd202::txCommand(char* ptr)
{
	int		n = strlen(ptr);
	char	buf[80];
	
	buf[0] = C_STX;							// packet header
	strcpy(&buf[1],ptr);					// command string
	buf[n+1] = C_ETX;							// end of command
	calChecksum(ptr,&buf[n+2]);				// checksum
	buf[n+4] = C_CR;							// CR  
	buf[n+5] = '\0';						// NULL terminate
	puts(buf);	           
}*/
/*
	<method> 		txCommand()
	<description>	Send command string
	<arguments>		const string str = string to display
	<return>		None
*/
void	CIsd202::txCommand(const string str, char *s)
{
	int		n = str.size() - 1;
	char	buf[80];
	char	dst[256];
	ISDST	st;///
	
	//eatPacket(st,"txCommand");
	ConsumeExcess();
	buf[0] = C_STX;							// packet header
	strncpy(&buf[1],str.c_str(),n);			// command string
	buf[n+1] = C_ETX;							// end of command
	calChecksum(str.c_str(),&buf[n+2]);		// checksum
	buf[n+4] = C_CR;							// CR  
	buf[n+5] = '\0';						// NULL terminate
	concats(dst,s,(char*)"txCmd2");
	puts(buf,0,dst);	           
}

/*
	Function 	: waitReady()
	Description	: Wait for operation panel Ready
	Arguments	: None
	Returns		:
*/
int		CIsd202::waitReady()
{
	int	retry;
	int	ret = false;
	ISDST	st;

	for(retry = 0; retry < 3; retry++) {
		txCommand("UZ");
		if(eatPacket(st,(char*)"wtRdy") == true){
			if(st.type == 'r'){
				ret = true;
				break;
			}
		}
	}
	if(ret == false)
		LOG("libPIF",LogErr,"Failed to talk to OP!");
	return(ret);
}
