/*
 	This file is part of Jenux.

    Jenux is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    Jenux is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with Jenux.  If not, see <http://www.gnu.org/licenses/>.

	Module			: CTDU.C
	Description		: Implementation of CTDU class. 
					  This class is a delived from CSerial class and
					  provides functionality to communicate with Operation 
					  panel ISD/202/ GOP-5084.
	Compiler		: gcc 4.8.5++
	Email			: matsuzawa.jei@nifty.com
	Maintenance		:
		Jul.13, '17	Released as GPL software officially
		Sep.26, '17	1) Added sound command
					2) Added backlight command
*/
#include 	<iostream>
#include 	<iomanip>
#include 	<fstream>
#include 	<sstream>
#include    <unistd.h>
#include    <sys/time.h>
#include    <sys/types.h>

using namespace std;

#include	"../../../include/pifg/libPIF.h"
#include 	"../../../include/pifg/common.h"
#include 	"../../../include/pif/CTDU.h"
#include 	"../../../include/pifg/CLog.h"
#include <../../../home/Genesis/include/pageengine.h>
#include "../../../include/cbib1830.hpp"
#include "../../../include/cbib.hpp"
#include </home/Genesis/include/lis.h>

GLOBALAREA*	   	globalarea;		// Global data area.

//	External classes ________________________________________________________
extern CLog	oLog;
int MBVersion;     // checking mother board version, Jetway(New) = 2, Gigabyte(Old) = 1

//extern int printdataOn;

//	External functions _____________________________________________________

static int	getTime();					// get current time in sec.

//  Local variables _________________________________________________________

const	int		Beep1[] = {30,60,30,0};
const	int		Beep2[] = {5,5,5,5,5,5,5,100,5,5,5,5,5,5,5,0};
const	int		Beep3[] = {5,5,5,5,5,0};
const	int*	Beep[] = { Beep1, Beep2, Beep3};

//		commands
const	char	write_start[] = "Ud";			// start of write mode
const	char	write_end[] = "Uc\r";			// end of write mode
const	char	page_start[] = "FS\r";			// start of page transfer 
const	char	page_end[] = "Fs\r";			// end of page transfer 
const	char	page_erase[] = "Fpageerase\r";	// page erase 
const	char	bmp_erase[] = "Fbiterase\r";	// bitmap data erase 
const	char	font_start[] = "FG\r";			// start of page transfer 
const	char	font_end[] = "Fg\r";			// start of page transfer 
const	char	font_erase[] = "Fgaijierase\r";	// font data erase 

const	char	get_csp[] = "Fb\r";				// get checksum of page data  
const	char	get_csf[] = "Fd\r";				// get checksum of flash rom  

const	char	ans_page_start[] = "Data Save Start";
const	char	ans_page_end[] = "Data Save End";
const	char	ans_page_erase[] = "PageData Erase Start";
const	char	ans_page_erase_end[] = "PageData Erase End";
const	char	ans_bmp_erase[] = "BitmapData Erase Start";
const	char	ans_bmp_erase_end[] = "BitmapData Erase End";
const	char	ans_font_start[] = "Gaiji Save Start";
const	char	ans_font_end[] = "Gaiji Save End";
const	char	ans_font_erase[] = "Gaiji Erase Start";
const	char	ans_font_erase_end[] = "Gaiji Erase End";

const	char	ans_mdw_start[] = "MDW_START";
const	char	ans_mdw_end[] = "MDW_END";

int     updateFlashFlag;

const	int		data_size = 256;

//	Global methods _________________________________________________________
/*
	Method		:	CTDU()
	Description	:	Constructor 
	Arguments	:	int fd = file descriptor
					bool model = 0: GOP-5084H/ 1: ISD-202
					int	skin = skin # (0: Gray/ 1: Green/ 2: Maroon/ 3: Navy)
*/
CTDU::CTDU(int fd, bool model, int skin)
{
	string lineFromFile;
    LOG("libPIF",LogInfo,"Creating CTDU class instance ...");
    ifstream mbVersion("/home/Genesis/motherBoardVersion.txt");
    if(mbVersion.is_open()){
        getline(mbVersion,lineFromFile); 
    MBVersion = atoi(lineFromFile.c_str());
    }
    nFd = fd;							// save a file descriptor
	bModel = model;						// save model
	nCrntSkin = skin;					// save skin
	port = 1;							// use channel-B

	pthread_mutex_init(&_ISDWaiterMutex,NULL);
	pthread_cond_init(&_ISDWaiterCond,NULL);
	
	putBaud(BAUD_38400);				// set baud rate 38400 bps
	putTrigger(TRIGGER_8);				// set RX trigger level to 8bytes 
    if(!waitReady())					// wait until panel gets ready
		nConnect = false;
	else
		initialize();
}
//		General methods
/*
	Function 	: KeyIntrOn()
	Description	: minus a key intr count
	Arguments	: None
	Returns		: None
*/
void CTDU::KeyIntrOn()
{
	if(_keyIntrCnt)
	{
		_keyIntrCnt--;
		if(_keyIntrCnt == 0)
			_keyIntrOn = 1;
	} else {
		printf("<<< OVER KeyIntrOn()!!! >>>\n");
	}
}

/*
	Function 	: KeyIntrOff()
	Description	: adds a key intr count
	Arguments	: None
	Returns		: None
*/
void CTDU::KeyIntrOff()
{
	_keyIntrOn = 0;
	_keyIntrCnt++;
}
/*
	Method	 	: disable()
	Description	: Disable panel
	Arguments	: None
	Return		: None
*/
void 	CTDU::disable()
{
	if(bModel)	// ISD-202
		txCommand("STOP");               	// stop panel
}

/*
	Method	 	: enable()
	Description	: Enable panel
	Arguments	: None
	Return		: None
*/
void 	CTDU::enable()
{
	if(bModel)	// ISD-202
		txCommand("RUN");               	// restart panel
}


//		Diplay-related methods __________________________________________
/*
	Method	 	: clearScreen()
	Description	: Clear all screen
	Arguments	: None
	Return		: None
*/
void	CTDU::clearScreen()
{
	ostringstream oss;
	char	buf[32];
	static int already = 0;
	
    if (bModel)
        nBG = ISD_WHITE;
    else
        nBG = GOP_WHITE;

	if(bModel){	// ISD-202
            sprintf(buf,"DE,%X",nBG);
            txCommand(buf);
	}
	else{		// GOP-5084
	oss << "CLR " << nBG;
	txCommand(oss.str());
	}
}

void	CTDU::clearScreen_New()
{
//	ostringstream oss;
	txCommand("WN Cmd0 C");
	txCommand("WN Cmd1 Q");
	txCommand("WN Cmd1");
}

void	CTDU::clearPage()
{
    ostringstream oss;
    
	if( _lastPage >= 3 && _lastPage <= 9 ) 
    {
		dispPage( _lastPage + 10 );	// put it back due to screen overlapped with ISD TDU
		dispPage( _lastPage - 10);
	} else if(_lastPage == 20) {	// New Menu w/ Select All
		dispPage( _lastPage + 10 );
		dispPage( _lastPage - 10);
	} 
	else 
        clearScreen();
}

/*
	Method		: display() - 1
	Description	: Display a character at specified position.
	Arguments	: int x = x address (column)
				  in y = y address (line)
				  char c = character
	Return		: None
*/
void	CTDU::display(int x, int y, char c, int bitmapFontNo)
{
	char buf[32];
    ostringstream oss;
    
	if(bModel){	// ISD-202
		sprintf(buf,"DO,%d,0,%d,%d,%X,%X,'%c",nSize,x,y,nFG,nBG,c);
        	txCommand(buf);
	}
	else {		// GOP-5084
    #if VGA // VGA
	oss << "TXT " << x * 2 << " " << y * 2 << " " << "2" << " 0 ";
    #else   // QVGA
	oss << "TXT " << x * 2 << " " << y * 2  << " 2 0 ";
    #endif
	oss << nFG << " " << nBG << " " << c; 
        txCommand(oss.str());
	}
	
}

/*
	Method	 	: display() - 3
	Description	: Display string at specified position.
	Arguments	: int x = x address (column)
				  int y = y address (line)
				  const string str = string
	Return		: None
*/
void	CTDU::display(int x, int y, const string str, int bitmapFontNo)
{
	char buf[1024];
    ostringstream oss;
    
        if(bModel){	// ISD-202
            sprintf(buf,"DO,%d,0,%d,%d,%X,%X,'%s",nSize,x,y,nFG,nBG,str.c_str());
            txCommand(buf);
        }
        else {		// GOP-5084
        #if	VGA    // VGA
            oss << "TXT " << x << " " << y << " " << " 2" << " 0 ";
        #else		// QVGA
            oss << "TXT " << x * 2 << " " << y * 2  << " 2 0 ";
        #endif
            oss << nFG << " " << nBG << " " << str; 
            txCommand(oss.str());
        }
}

/*
	Method		: display() - 4
	Description	: Display string at specified position.
	Arguments	: const POINT& p = reference to POINT 
				  const string str = string
	Return		: None
*/
void	CTDU::display(const POINT& p, const string str,int bitmapFontNo)
{
    char buf[1024];
    ostringstream oss;

    
	if(bModel){	// ISD-202
	if(*str.c_str())
        {
            sprintf(buf,"DO,%d,0,%d,%d,%X,%X,'%s",nSize,p.x,p.y,nFG,nBG,str.c_str());
            txCommand(buf);
        }
	}
	else {		// GOP-5084
    #if VGA 	// VGA	
        oss << "TXT " << p.x << " " << p.y << " " << " 2 " << " 0 ";
    #else		// QVGA
        oss << "TXT " << p.x * 2 << " " << p.y * 2  << " 2 0 ";
    #endif
        oss << nFG << " " << nBG << " " << str; 
        txCommand(oss.str());
	}
	
}

/*
	Method		: display() - 2
	Description	: Display string at specified position.
				  String must be terminated by NULL.
	Arguments	: int x = x address (column)
				  int y = y address (line)
				  const char* str = pointer to string
	Return		: None
*/

void	CTDU::display(int x, int y, const char* str, int bitmapFontNo)
{
	char buf[1024];
	ostringstream oss;
            if(bModel){	// ISD-202
                sprintf(buf,"DO,0,0,%d,%d,%X,%X,'%s",x,y,nFG,nBG,str);
                txCommand(buf);
            }
            else {		// GOP-5084
                oss << "WN Cmd0 T" << x*2 << "," << y*2 << ",";
                oss << "42,32768,"<< str; 
                txCommand(oss.str());
		txCommand("WN Cmd1 Q");
                txCommand("WN Cmd 1");
           	usleep(300000);         //delay 100ms
            } 
}

void	CTDU::displayRev(int x, int y, char *str, int rev, int bitmapFontNo)
{
	char	buf[80];
    ostringstream oss;

//	if(str)
//	{
//		if(*str)
//		{
            if(bModel)
            {
                if( rev ) 
                    sprintf(buf,"DO,0,0,%d,%d,%X,%X,'%s",x,y,nBG,nFG,str);
                else 
                    sprintf(buf,"DO,0,0,%d,%d,%X,%X,'%s",x,y,nFG,nBG,str);
                txCommand(buf);
            }
          else if(!bModel)
            {
                if( rev )
                     oss << "TXT " << x * 2 << " " << y * 2  << " 2 0 " << nBG << " " << nFG << " " << str;
                else
                oss << "TXT " << x * 2 << " " << y * 2  << " 2 0 " << nFG << " " << nBG << " " << str;
                txCommand(oss.str());
            } 
		}

void	CTDU::displayRevCustomColor(int x, int y, char *str, int rev, int fgColor, int bgColor, int bitmapFontNo)
{
	char	buf[1024];
    ostringstream oss;
            if(bModel)
            {
                if( rev ) 
                    sprintf(buf,"DO,0,0,%d,%d,%X,%X,'%s",x,y,bgColor,fgColor,str);
                else 
                    sprintf(buf,"DO,0,0,%d,%d,%X,%X,'%s",x,y,fgColor,bgColor,str);
                txCommand(buf);
            }
            else
            {
                if( rev )
                    oss << "TXT " << x * 2 << " " << y * 2  << " 2 0 " << bgColor << " " << fgColor << " " << str;
                else
                    oss << "TXT " << x * 2 << " " << y * 2  << " 2 0 " << fgColor << " " << bgColor <<  " " << str;
                txCommand(oss.str());
            }            
		}

/*
	Method	 	: dispPage()
	Description	: Disply a page
	Arguments	: int page = page#
				  bool flag = 0: clear screen before displaying page (GOP-5084H)
	Return		: None
*/
void 	CTDU::dispPage(int page, bool flag)
{
    char	buf[16];
	*buf = '\0';
	_lastPage = page;
    
   	if(bModel){ 					// ISD-202
 //     if(!flag)
//		clearScreen();             // this one need to comment out for screen flashing 2,3 times in error screen and every state changed 									goes away, for ISD TDU, uncomment is woking for overlapping but screen refresh 2,3 times, will look into later
	sprintf(buf,"MWS,Page,%d",page);
        txCommand(buf);
	}
	else {//if(page != nCrntPage){		// GOP-5084
		if(!flag)
			clearScreen();
        sprintf(buf,"WN PAGE %d",page);
   		txCommand(buf);
		if(!nBL){
			backlight(1);			// backlight on
			nBL = 2;
		}
	}
	nCrntPage = page;
}

/*
	Method	 	: drawBox()
	Description	: Draw box
	Arguments	: POS& pos = (x1,y1),(x2,y2)
				  int c = color
	Return		: None
*/
void	CTDU::drawBox(POS& pos, int c)
{

    char	buf[32];
    *buf = '\0';
    
    if(bModel)	// ISD-202
		sprintf(buf,"DB,%d,%d,%d,%d,%d,%d",pos.x1,pos.y1,pos.x2,pos.y2,c,c);
	else	// GOP-5084
		sprintf(buf,"RECT %d %d %d %d %d %d",pos.x1*2,pos.y1*2,pos.x2*2,pos.y2*2,c,c);
	txCommand(buf);
	
}

void CTDU::drawBox(int x1, int y1, int x2, int y2, int c)
{
	char	buf[32];
    *buf = '\0';
    
    if(bModel)	// ISD-202
        sprintf( buf, "DB,%d,%d,%d,%d,%d,%d", x1, y1, x2, y2, c, c );
    else
        sprintf( buf, "RECT %d %d %d %d %X %X", x1*2, y1*2, x2*2, y2*2, c, c );
    txCommand(buf);

	return;
}

void CTDU::drawCustomLine(int x1, int y1, int x2, int y2, int frameColor)
{
	char	buf[32];
    *buf = '\0';
    
    if(bModel)	// ISD-202
        sprintf( buf, "DL,%d,%d,%d,%d,%X", x1, y1, x2, y2, frameColor );
    else
        sprintf( buf, "DL %d %d %d %d %X", x1*2, y1*2, x2*2, y2*2, frameColor );
	txCommand(buf);

	return;
}

void CTDU::drawDot(int x, int y, int c)
{
	char	buf[32];
    *buf = '\0';
    
    if(bModel)	// ISD-202
        sprintf( buf, "DD,%d,%d,%X", x, y, c );
    else
        sprintf( buf, "DD %d %d %X", x*2, y*2, c );
    txCommand(buf);
}

void CTDU::drawLine(int x1, int y1, int x2, int y2, int c)
{
	char	buf[32];
    *buf = '\0';
    
    if(bModel)	// ISD-202
        sprintf( buf, "DL,%d,%d,%d,%d,%X", x1, y1, x2, y2, c );
    else
        sprintf( buf, "DL %d %d %d %d %X", x1*2, y1*2, x2*2, y2*2, c );
	txCommand(buf);
}

void CTDU::drawRect(int x1, int y1, int x2, int y2, int c)
{
	drawLine( x1, y1, x2, y1, c );
	drawLine( x1, y1, x1, y2, c );
	drawLine( x2, y1, x2, y2, c );
	drawLine( x1, y2, x2, y2, c );
}

void CTDU::drawPicture(int x, int y, int ndx)
{
	char	buf[32];
    ostringstream oss;
    
    if(bModel){
        sprintf( buf, "DI,%d,%d,%d", x, y, ndx );
        txCommand(buf);
    }
    else{
    oss << "BMP " << x * 2 << " " << y * 2  << " " << ndx << " " << 32768 << " 0 0";
    txCommand(oss.str());
    }
	
}

void CTDU::drawEclipse(int cx, int cy, int rx, int ry, int fill_col, int frame_col)
{
	char	buf[32];

	sprintf( buf, "DC,%d,%d,%d,%d,%d,%d", cx, cy, rx, ry, fill_col, frame_col );
	txCommand(buf);
}

void CTDU::drawEclipse2(int x1, int y1, int x2, int y2, int fill_col, int frame_col)
{
	char	buf[32];

	sprintf( buf, "DC,%d,%d,%d,%d,%d,%d", (x1+x2) >> 1, (y1+y2) >> 1, abs(x2-x1) >> 1, abs(y2-y1) >> 1, fill_col, frame_col );
	txCommand(buf);
}

void CTDU::drawCircle(int cx, int cy, int r, int fill_col, int frame_col)
{
	char	buf[32];

	sprintf( buf, "DC,%d,%d,%d,%d,%d,%d", cx, cy, r, r, fill_col, frame_col );
	txCommand(buf);
}

/*
	Method	 	: setColor()
	Description	: Set Color for testr
	Arguments	: int fg = foreground color
				  int bg = background color
	Return		: None
*/
void	CTDU::setColor(int fg, int bg)
{
//	nFG = colorTbl[fg][bModel];
//	nBG = colorTbl[bg][bModel];
    nFG = fg;
    nBG = bg;
}

void CTDU::DrawClock(int xx, int yy, int rad, int rev, int rev2, int col, int col2)
{
	int x, y;
	static int cc=0;
	
	//drawCircle( xx, yy, 10, nBG, nFG );
	//drawCircle( xx, yy, 10, 0, nFG );
   	switch(rev)
   	{
   		case 0:	 x = 0;	 y = -5; break;
   		case 1:	 x = 3;	 y = -4; break;
   		case 2:	 x = 4;	 y = -3; break;
   		case 3:	 x = 5;	 y = 0; break;
   		case 4:	 x = 4;	 y = 3; break;
   		case 5:	 x = 3;	 y = 4; break;
   		case 6:	 x = 0;	 y = 5; break;
   		case 7:	 x = -3; y = 4; break;
   		case 8:	 x = -4; y = 3; break;
   		case 9:	 x = -5; y = 0; break;
   		case 10: x = -4; y = -3; break;
   		case 11: x = -3; y = -4; break;
   	}
   	if( (cc & 1) == 0 ) {
	   	if( ((cc/24)%2) == 1 )
			drawLine( xx, yy, xx + (x * 2), yy + (y * 2), col );
		else
			drawLine( xx, yy, xx + (x * 2), yy + (y * 2), /*nBG*/col2 );
	} else {
		drawLine( xx, yy, xx + (x * 2), yy + (y * 2), nBG );
	}
	cc++;
   	switch(rev2)
   	{
   		case 0:	 x = 0;	 y = -5; break;
   		case 1:	 x = 3;	 y = -4; break;
   		case 2:	 x = 4;	 y = -3; break;
   		case 3:	 x = 5;	 y = 0; break;
   		case 4:	 x = 4;	 y = 3; break;
   		case 5:	 x = 3;	 y = 4; break;
   		case 6:	 x = 0;	 y = 5; break;
   		case 7:	 x = -3; y = 4; break;
   		case 8:	 x = -4; y = 3; break;
   		case 9:	 x = -5; y = 0; break;
   		case 10: x = -4; y = -3; break;
   		case 11: x = -3; y = -4; break;
   	}
	//drawLine( xx, yy, xx + (x * 2), yy + (y * 2), col2 );

	if( ++rev >= 12 ) {
		rev = 0;
		if( ++rev2 >= 12 ) {
			rev2 = 0;;
		}
	}
	
}

void CTDU::drawColorMatrix()
{
	for(int b=0; b < 8; b++) {
   	for(int a=0; a < 8; a++) {
		drawBox( 20*a, 		15*b, 		20*a+19, 		15*b+14, 		((7-b) << 5)+((7-a) << 2) + 0);
		drawBox( 160 + 20*a,15*b, 		160 + 20*a+19,	15*b+14, 		((7-b) << 5)+(a << 2) + 1);
		drawBox( 20*a, 		120 + 15*b, 20*a+19, 		120 + 15*b+14,	(b << 5)+((7-a) << 2) + 3);
		drawBox( 160 + 20*a,120 + 15*b, 160 + 20*a+19,	120 + 15*b+14,	(b << 5)+(a << 2) + 2);
		///drawBox( 20*a, 16*b, 20*a+19, 16*b+15, (((b*16+a) & 7) << 5) + ((((b*16+a) >> 4)  & 7) << 2)+((b>>2) & 2) + ((a>>3) & 1));
   	}
    }
}

int	CTDU::getKey_FWFlash(GOPST& st)
{
    while(isEmpty()){}
	return(eatPacket(st));
}

//	Key input methods
/*
	Method	 	: getKey()
	Description	: Wait for key
	Arguments	: ISDST& st = reference to ISDST
	Return		: int true = accepted
					  false = error
*/
int	CTDU::getKey(GOPST& st)
{
    int result;
	int rate = 20;
		
    	while(1)
   	{
		if(_keyLocked)
		{

		} else 
		{
		if(!isEmpty((char*)"getKey:")) {
                	if (updateFlashFlag == 1)
                    	    printf("getKey: Break;");
                	else
                    	    break;
        }
        }
    }
    pthread_cond_signal(&_ISDWaiterCond);
    _keyPresses++;
    result = eatPacket(st,(char*)"gtKy");
	if(st.type) {
		_st.type = st.type;
		_st.cmd = st.cmd;
		_st.par = st.par;
	}
    
	return result;
}

void CTDU::waitKey()
{
	pthread_mutex_lock(&_ISDWaiterMutex);
	pthread_cond_wait(&_ISDWaiterCond,&_ISDWaiterMutex);
	pthread_mutex_unlock(&_ISDWaiterMutex);
}

/*
	Function 	: WaitKey()
	Description	: Wait for key
	Arguments	: int key = key code  (0: any keys)
				  int flag = 0 : wait forever
				  			 1 : return immediately
	Returns		: int true = specified key is pressed
*/
int	CTDU::waitKey(int key, int flag)
{
	GOPST	st;

	while(true) {
		if(isEmpty((char*)"waitKey:") == false){
			if(eatPacket(st,(char*)"wtKy")){
				if((key == st.cmd) || !key)
					break;
				else if(st.cmd == 'R')
					return false;
			}
			else
				return false;
		}
		else if(flag == 1)
			return false;
	}
	return true;
}

int CTDU::waitKeyPresses()
{
	int rate = 20;
	while(!_keyPresses && !_redisplayMenu)
	{
	if(MBVersion == 2)
            usleep(1000000/rate);
    	else if(MBVersion == 1)
            usleep(1000);
   //         usleep(10000);
	}
	if ((MBVersion ==1) && (_keyPresses))
        usleep(10000);   // added for menu delay response in old motherboard,version 60021
     //       usleep(15000);  // this is also a good value to test 
    
	_keyPresses=0;
}

int CTDU::isKeyPressed() {
	return _st.type;
}

/*int CIsd202::experiment(ISDST *st) {
//int CIsd202::experiment() {
	return 0;
}*/
int CTDU::getKeyPressed() {
	int key;
	if( _st.type == 'A' ) {
		key = _st.cmd;
		//printf("key=%d, %x\n", key, key );
		_st.type = _st.cmd = _st.par = 0;
	/*	switch(key) {
			
			default:
				key = key;
		}*/
		return key;
	} else {
		return 0;
	}
}

int	CTDU::getLastKey(GOPST& st) {			// get last key code
	if(st.type) {
		st.type = _st.type;
		st.cmd = _st.cmd;
		st.par = _st.par;
	}
	if( _st.type == 'A' ) {
		///_st.type = _st.cmd = _st.par = 0;
		return 1;
	} else {
		return 0;
	}
}

//	Sound-related methods ___________________________________________
/*
	Method	 	: beeper()
	Description	: beep controller
	Arguments	: int pattern = pattern#
	Return		: None
*/
void 	CTDU::beeper(const int pattern)
{
	int	i,j;
	int const*	ptr = Beep[pattern];

	j = 1;
	while((i = *ptr++) != 0){
		if(j & 1)
			buzzer(i);
		usleep(i * 1000);
		j++;
	}
}

/*
	Method	 	: buzzer() - 1
	Description	: Ring the buzzer at the specified time
	Arguments	: int ms = active time in ms
	Return		: None
*/
void	CTDU::buzzer(const int ms)
{
	char	buf[8];
	if(bBuzzer){
		if(ms > 0){
			if(bModel)	// ISD-202
				sprintf(buf,"DR,%d",ms);   
			else		// GOP-5084
				sprintf(buf,"WD wF090 %d",ms);  
			txCommand(buf);
		}
	}
}

/*
	Method	 	: buzzer() - 2
	Description	: Ring the buzzer intermittently
	Arguments	: int on = active time in ms
				  int off = inactive time in ms
	Return		: None
*/
void	CTDU::buzzer(const int on, const int off)
{
	char	buf[16];

	if(bBuzzer){
		if(bModel){	// ISD-202
			sprintf(buf,"dr2,%d,%d",on,off);  
            txCommand(buf);
        }
		else {		// GOP-5084
    		sprintf(buf,"WN BEEPINTERVAL %d",on+off);    
			txCommand(buf);
            *buf = '\0';
			if(on+off)
                sprintf(buf,"WN BEEPLONG %d",on);
			else
                sprintf(buf,"WN BEEPLONG 3");
            txCommand(buf);
		}
	}
}

//	Memory access methods ___________________________________________________

/*
	Method	 	: readMemory()
	Description	: Read a value in the panel memory
	Arguments	: int idx = index of memory 
	Return		: int val = value
*/
void
CTDU::readMemory(int idx)
{
	ostringstream oss;
	GOPST	st;
	int		val = -1;
	char	buf[32];
	string	s;

	if(bModel){	// ISD-202
		sprintf(buf,"MR,MS%d",idx);
		txCommand(oss.str());                			// send read command
		if(eatPacket(st) == true){     					// Rx reply
			if(st.type == 'R')
				val = st.par;
		}
	}
	else{		// GOP-5084
        oss << "RN MS" << setw(3) << setfill('0') << idx;
		txCommand(oss.str());                			// send read command
	}
}

/*
	Method	 	: writeMemory()
	Description	: Write value in the panel memory
	Arguments	: int idx = index of memory 
				  int val = value
	Return		: None
*/
void	CTDU::writeMemory(int idx, int val)
{
	ostringstream oss;

	if(bModel)	// ISD-202
		oss << "MWS,MS" << idx << "," << val;
	else		// GOP-5084
		oss << "WN MS" << setw(3) << setfill('0') << idx << " " << val;
	txCommand(oss.str());
}

/*
    Method      : writeMemory()
    Description : Write text string in the panel memory
    Arguments   : int idx = index of memory
                  char* ptr = pointer to text string
    Return      : None
*/
void    CTDU::writeMemory(int idx, char* ptr)
{
    char    buf[64];
    if (bModel){
        sprintf(buf,"MWT,MT%d,'%s",idx,ptr);
        txCommand(buf);
    }
}    

/*
	Method		: getChecksum()
	Description	: Get checksum of ISD202
	Arguments	: int ctrl = 0: all area 
							 1: imagearea 
	Return		: None
*/
int	CTDU::getChecksum(int ctrl)
{
	char	buf[16];
	int	checksum = 0;

	LOG("libPIF",LogInfo,"Start write mode");
	txCommand(write_start);		// start write mode
	sleep(3);					// wait for reboot (3 sec.)

	if(!ctrl){
        if (bModel)
            puts(get_csf,0);			// get checksum of whole flash rom
        else
            puts("Sd",0);
    }
	else if(ctrl == 1)
		puts(get_csp,0);			// get checksum of image area 
	else
		return checksum;
	if(rxAnswer(buf,5))
		cout << "> checksum = " << buf << endl;
	buf[2] = '\0';
	sscanf(buf,"%x",&checksum);	
    
    LOG("libPIF",LogInfo,"End write mode");
	puts(write_end,0,(char*)"gtChS3");	

	return checksum;
}

//	Flashing data ___________________________________________________
/*
	Method		: updateFlash()
	Description	: Update flash memory on the ISD-202
	Arguments	: char* fname = pointer to filename
				  int flag = 1: graphics data exists
	Return		: None
*/
int	CTDU::updateFlash(char* fname, int flag)
{
	char	buf[80];
	char	lbuf[120];
	
    updateFlashFlag = 1;
	sprintf(buf,"Open page data file: %s",fname);
	LOG("libPIF",LogInfo,buf);
	ifstream	fin(fname);
	if(!fin){
		sprintf(buf,"Cannot open a file: %s",fname);
		LOG("libPIF",LogErr,buf);
		return false;
	}

	LOG("libPIF",LogInfo,"Start write mode");
	txCommand(write_start);		// start write mode
	sleep(3);					// wait for reboot (3 sec.)

	puts(page_start,0);			// start page transfer mode
	if(rxAnswer(buf,5)){
		if(strcmp(buf,ans_page_start))
			return false;
		else{
			sprintf(lbuf,"> %s",buf);
			LOG("libPIF",LogInfo,lbuf);
		}
	}
	else
		false;

	puts(page_erase,0);				//  page erasing ... 
	if(rxAnswer(buf,5)){
		if(strcmp(buf,ans_page_erase))
			return false;
		else{
			sprintf(lbuf,"> %s",buf);
			LOG("libPIF",LogInfo,lbuf);
		}
	}
	else
		false;
	if(rxAnswer(buf,30)){
		if(strcmp(buf,ans_page_erase_end))
			return false;
		else{
			sprintf(lbuf,"> %s",buf);
			LOG("libPIF",LogInfo, lbuf);
		}
	}
	else
		return false;

	if(flag){
		puts(bmp_erase,0);				//  bitmap erasing ... 
		if(rxAnswer(buf,5)){
			if(strcmp(buf,ans_bmp_erase))
				return false;
			else{
				sprintf(lbuf,"> %s",buf);
				LOG("libPIF",LogInfo, lbuf);
			}
		}
		else
			return false;
		if(rxAnswer(buf,30)){
			if(strcmp(buf,ans_bmp_erase_end))
				return false;
			else{
				sprintf(lbuf,"> %s",buf);
				LOG("libPIF",LogInfo,lbuf);
			}
		}
		else
			return false;
	}

	while(!fin.eof()){
		fin.getline(buf,77,C_LF);
		strcat(buf,"\n");
		puts(buf,0);	
		if((buf[0] == 'B') || (buf[0] == 'P')){
			if(rxAnswer(buf,1)){
				sprintf(lbuf,"> %s",buf);
				LOG("libPIF",LogInfo, lbuf);
			}
		}
	}
	fin.close();

	puts(page_end,0);				// end page transfer mode
	if(rxAnswer(buf,5)){	
		if(strcmp(buf,ans_page_end))
			return false;
		else{
			sprintf(lbuf,"> %s",buf);
			LOG("libPIF",LogInfo,lbuf);
		}
	}
	else
		return false;
	LOG("libPIF",LogInfo,"End write mode");
	puts(write_end,0);			// end write mode
	sleep(3);
	updateFlashFlag = 1;
	return	true;
}

/*
	Method		: restart()
	Description	: initialize TDU 
	Arguments	: None
	Return		: None
*/
void
CTDU::restart()
{
	initialize();
}

/*
	Method		: updateGOP()
	Description	: Update flash memory on the GOP-5084H
	Arguments	: const char* idx = pointer to index file 
	Return		: None
*/

int
CTDU::updateGOP(const char* idx)  // idx = /home/jenux/bin/index.txt
{
	ifstream in;
	string	s;
    
    updateFlashFlag = 1;
	upCommand("STOP");
	sleep(1);
//	delCommand("STOP");
	txCommand("CLEAR");

	in.open(idx);
	if(in.fail()){
		cout << idx << " does not exist!\n";
		return false;
	}
	while(getline(in,s)){
	if(!upFile(s))	// s= /home/jenux/shared/uploads/gop/GB00/*.7z
		return false;
	}
//	upCommand("REWRITE");	// this need to comment out to run the genesis right after the panel flash,
				// if not the panel flash routine is waiting the user to touch the screen.
	sleep(1);
//	delCommand("REWRITE");
	updateFlashFlag = 1;
	upCommand("RESET");
	cout << "after Reset upCommand" << endl;
	sleep(15);
//	upCommand("REWRITE");
	return true;
}

/*
	Method		: restart()
	Description	: initialize TDU 
	Arguments	: None
	Return		: None

void
CTDU::restart()
{
	initialize();
}*/

// Private methods ________________________________________________________

/*
	Method 		: initialize()
	Description	: Initialize variables in CTDU class
	Arguments	: None
	Return		: None
*/
void
CTDU::initialize()
{
	nConnect = true;
//	if(bModel)	
//		nBG = 0 + (0 <<3) + (0 << 6) + 4 + 32;					// default foregroung text color
    clearScreen();					// clear screen
	if(bModel){
        nBG = 0 + (0 <<3) + (0 << 6) + 4 + 32;
        nFG = ISD_BLACK;					// default foregroung text color
        nBG = ISD_WHITE;                   	// default background text color
    }else{
        nFG = GOP_BLACK;					// default foregroung text color
        nBG = GOP_WHITE;                   	// default background text color
    }
    
	nSize = S_NORMAL;				// default character size
	bBuzzer = true;					// error buzzer active
	_keyLocked=0;
	_keyIntrOn=1;
	_keyIntrCnt=0;
	_keyPresses=0;
	_redisplayMenu=0;
	_menuInside = 0;
	bMode = 0;
	nCrntPage = 0;
	version();						// get firmware version
	if(!bModel){					// GOP-5084H
		checksum();						// het checksum of TDU data
        txCommand("WN BEEPLONG 3");	// default buzzer
		backlight(1);				// backlight on
	}
}

void	CTDU::calChecksum(const char* src, char* dest)
{
	int		checksum = 0;
	int		c;
	
	while((c = (int)*src) != '\0'){
		checksum ^= c & 0xff;
		src++;
	}  
	sprintf(dest,"%02x",checksum);	   
}

void CTDU::ConsumeExcess()
{
	while( !isEmpty((char*)"ConsumeExcess:") ) {
		getchar(0,(char*)"ConEx");								// Rx trailing characters
	}
}

/*
	Method 		: eatPacket()
	Description	: Analyse command packet
	Arguments	: ISDST &cmd = reference to ISDST structure
	Return		: int true = normal completion
					  false = Rx Error
*/
int CTDU::eatPacket(GOPST& st, char *s)
{
	int		i,j,c;
	int		par;
	char	tmp[32];
	char	csbuf[4];
	char	csbuf2[4];
	int		checksum = 0;
	int		cscount = false;
	char	dst[256];
	char	str[256];
	int		lid;

	concats(dst,s,(char*)"etPk");
	lid=BeginCmdSeq(dst);///
	sprintf(str,"etPk[%d]",lid);
	concats(dst,s,str);
	j = 0;
	i = 0;
	st.type = 0;
	st.cmd = 0;
	st.par = 0;

	while(true) {
		if((c = getchar(0,dst)) == C_STX){		// Hunting STX code ...
			cscount = true;				// Start to calc. checksum
			break;					// Break if STX
		}
		else if(c == false){
			ConsumeExcess();
			EndCmdSeq(lid,dst);
            return false;						// Return if Rx Error
        }
        else{ 
			printf("WOAH! WOAH! WOAH! WOAH! WOAH! WOAH!\n");
			printf("WOAH! WOAH! WOAH! WOAH! WOAH! WOAH!\n");
			printf("WOAH! WOAH! WOAH! WOAH! WOAH! WOAH!\n");
			printf("WOAH! WOAH! WOAH! WOAH! WOAH! WOAH!\n");
			printf("WOAH! WOAH! WOAH! WOAH! WOAH! WOAH!\n");
			printf("WOAH! WOAH! WOAH! WOAH! WOAH! WOAH!\n");
			printf("WOAH! WOAH! WOAH! WOAH! WOAH! WOAH!\n");
			cscount = true;
			tmp[i++] = c;					// store RX data
			checksum ^= c;					// calculate checksum
			break;
		}
	}
	for(i = 0; i < 32; i++) {
		c = getchar(0,dst);				// Rx next character
		if(c == false){
			ConsumeExcess();
            EndCmdSeq(lid,dst);
            return false;						// Return if Rx Error
        }
		if(c == C_LF){
			csbuf[j] = '\0';
			break;
		} 						// break if LF
		if(c == C_ETX){
			cscount=false; 		// Stop to calc. checksum ETX
			j = 0;
			tmp[i] = '\0';
		}
		else if (c >= ' '){
			if(cscount == true){
				tmp[i] = c;				// store RX data
				checksum ^= c;			// calculate checksum
			}
			else if(j < 2)
				csbuf[j++] = c;
		}
	}
	if(c != C_LF){				// Return if packet broken
		ConsumeExcess();
        EndCmdSeq(lid,dst);
        return false;
    }
	sprintf(csbuf2,"%02x",checksum);
    
	if(strcmp(csbuf,csbuf2)){
		printf("Checksum error (%s) (%s) [%s]\n",csbuf,csbuf2,tmp);
		ConsumeExcess();
        EndCmdSeq(lid,dst);
        return false;			// Return if check sumerror
	} else {
	}
	st.type = tmp[0];			// 1st byte = type
	switch(st.type){
		case 'r':	break;
		case 'A':	        	// memory action
			st.cmd = tmp[2];
			st.par = 0;
			i = 3;
			while(tmp[i])
				st.par = st.par * 10 + tmp[i++] - '0';
			break;
		case 'R':				// reply
			st.par = getVal(tmp);
			break;
        case 'G':	
        case 'I':	
            st.str= (char*)tmp;
			break;
	}							// Rx trailing characters
    ConsumeExcess();
    EndCmdSeq(lid,dst);
	return true;
}

/*
	Method	 	: getVal()
	Description	: Get value in the replied string
   	Arguments	: char* ptr = pointer to string
	Return		: int val
*/
int	CTDU::getVal(char* ptr)
{
	int		val = 0;
	int		i;

	while(*ptr++ != '='){}		// hunt '='
	while(*ptr){
		if((val >= '0') && (val <= '0'))
			val = (val * 10) + (*ptr++ - '0');
		else
			ptr++;
	}
	return val;
}

/*
	Method 		: rxAnswer()
	Description	: Receive answer form ISD-202
	Arguments	: char* buf = pointer to buffer
				  int timeout = time out (sec.)
	Return		: int true = normal completion
					  false = Time out
*/
int		CTDU::rxAnswer(char* buf, int timeout)
{
	int	c;
	char*	buf2 = buf;
	int	start = getTime();						// start timer

	while((c = getchar(0,(char *)"rxAn1")) != C_LF){
		if(c){
            cout << (char) c;
        if(c != C_CR)
            *buf++ = c;
        }
		else if(abs(getTime() - start) > timeout){
			cout << "O.P Rx Time Out !" << endl;
            *buf = '\0';
            getchar(0,(char *)"rxAn2");
			return false;				// return if error
		}
	}
	*buf = '\0';
	if(bModel)		//ISD
        	getchar(0,(char *)"rxAn2");
	return true;
}

/*
	Method 		: rxValue()
	Description	: Receive answer form GOP-5084
	Arguments	: string s = RX message
	Return		: int = returned value 
*/
int		CTDU::rxValue(string s)
{
	int		val = 0;
	stringstream ss;
	int		from,to;

	from = s.find("=") + 1;
	to = s.find("@") - from;
	ss << s.substr(from,to);
	ss >> val;
	cout << "val = " << val << endl;
	return val;
}

/*
	<method> 		txCommand()
	<description>	Send command string
	<arguments>		const char* ptr = pointer to string
	<return>		None
*/
void	
CTDU::txCommand(const char* ptr, char *s)
{
	int		n = strlen(ptr);
	char	buf[256];
	char	dst[256];
	char	str[256];

	_scratchStatus=0;
	buf[0] = C_STX;	
	_scratchStatus=1;
	strcpy(&buf[1],ptr);					// command string
	_scratchStatus=2;
	buf[n+1] = C_ETX;						// end of command
	_scratchStatus=3;
	calChecksum(ptr,&buf[n+2]);				// checksum
	_scratchStatus=4;
	buf[n+4] = C_CR;						// CR  
	_scratchStatus=5;
	buf[n+5] = '\0';						// NULL terminate
	_scratchStatus=6;
	_scratchStatus=7;
	sprintf(str,"txCmd1[%d,%d] = --<%s>--\n",strlen(buf),strlen(ptr),ptr);
	concats(dst,s,str);
	_scratchStatus=8;
	puts(buf,0,dst);
	_scratchStatus=9;
}

/*
	<method> 		txCommand()
	<description>	Send command string
	<arguments>		const string str = string to display
	<return>		None
*/
void	CTDU::txCommand(const string str, char *s)
{
//	int		n = str.size() - 1;
	int		n = str.size();
	char	buf[128];
	char	dst[256];
	GOPST	st;///
	
	if(n){
		ConsumeExcess();		
		cout << str << endl;
		buf[0] = C_STX;					// packet header
		strcpy(&buf[1],str.c_str());	// command string
		buf[n+1] = C_ETX;				// end of command
		calChecksum(str.c_str(),&buf[n+2]);	// checksum
		buf[n+4] = C_CR;					// CR  
		buf[n+5] = '\0';					// NULL terminate
		concats(dst,s,(char*)"txCmd2");
		puts(buf,0,dst);	           
	}
	else
		cout << "!" << endl;
}

/*
	<method> 		txCommand()
	<description>	Send command string
	<arguments>		string str = string to display
					int n = string length
	<return>		None
*/
void	CTDU::txCommand(char* str, int n)
{
	int		checksum = 0;
	char	buf[data_size+8];
	char	c;
	
	buf[0] = C_STX;							// packet header
	for(int i = 1; i <= n; i++){
		buf[i] = c = *str++;
            checksum ^= (c & 0xff);
	}
	buf[n+1] = C_ETX;							// end of command
	buf[n+2] = checksum;					// end of command
	buf[n+3] = C_CR;							// CR 
	putn(buf,n+4);
}

/*
	Method	 	: waitReady()
	Description	: Wait for operation panel Ready
	Arguments	: None
	Return		:
*/
int		CTDU::waitReady()
{
	int	retry;
	int	ret = false;
	GOPST	st;

	if(bModel){				// ISD_202
    for(retry = 0; retry < 3; retry++) {
        txCommand("UZ",(char*)"");
        if(eatPacket(st,(char*)"wtRdy") == true){
            if(st.type == 'r'){
                ret = true;
                break;
            }
        }
    }
		if(ret == false)
			LOG("libPIF",LogErr,"Failed to talk to OP!");
	}
	else
		ret = true;
	return(ret);
}

/*
	<method>		upCommand
	<description>	send file name to be uploaded	
	<args>			const char* str = pointer to file name 
	<return>	 	bool = true: OK
						   false: error	
*/
bool
CTDU::upCommand(const char* str)
{
	char	buf[8];
	ostringstream oss;

	oss << "UPLOAD " << str;
	txCommand(oss.str());
	rxAnswer(buf,5);
	if(buf[0] == NAK){
		cout << str << " already exist!\n";
		return false;
	}
	else if(buf[0] != ACK)
		return false;
	oss.str("");
	oss << "ULE";
	txCommand(oss.str());
	rxAnswer(buf,5);
	if(buf[0] != ACK)
		return false;
	return true;	
}

/*
	<method>		delCommand
	<description>	send file name to be deleted	
	<args>			const char* str = pointer to file name 
	<return>	 	bool = true: OK
						   false: error	
*/
bool
CTDU::delCommand(const char* str)
{
	char	buf[8];
	ostringstream oss;

	oss << "DELETE " << str;
	txCommand(oss.str());
	rxAnswer(buf,5);
	if(buf[0] == NAK){
		cout << str << " does not exist!\n";
		return false;
	}
	else if(buf[0] != ACK)
		return false;
	return true;	
}

/*
	<method>		upFile
	<description>	upload file	
	<args>			string file = file name 
	<return>	 	bool = true: OK
						   false: error	
*/
bool
CTDU::upFile(string file){
	ostringstream oss;
	ifstream in;
	char	buf[data_size + 4];
	int		idx,size;
	int		addr = 0;
	
	in.open(file.c_str(),ios::binary);
	if(in.fail()){
		cout << file << "is not found!\n";
		return false;
	}	
	idx = file.find("/gop/");
	if(!idx)
		return false;
	size = in.seekg(0, ios::end).tellg();
	in.seekg(0, ios::beg);
	oss << "FASTUPLOAD " << file.substr(idx+5);
	txCommand(oss.str());
	rxAnswer(buf,5);
	if(buf[0] != ACK)
		return false;

//	send file size
	buf[0] = size & 0xff; 
	buf[1] = (size >> 8) & 0xff; 
	buf[2] = (size >> 16) & 0xff; 
	buf[3] = (size >> 24) & 0xff; 
	txCommand(buf,4);
	rxAnswer(buf,5);
	if(buf[0] != ACK)
		return false;

//	send binary data
	do{
		buf[0] = addr & 0xff; 
		buf[1] = (addr >> 8) & 0xff; 
		buf[2] = (addr >> 16) & 0xff; 
		buf[3] = (addr >> 24) & 0xff; 
		memset(&buf[4],0,data_size);
		in.read(&buf[4],data_size);
		txCommand(buf,data_size+4);
        rxAnswer(buf,5);
        if(buf[0] != ACK)
            return false;
		int n = in.gcount();
		size -= n;
		addr += n;
	}while(size);
    
//	rxAnswer(buf,5);
//	if(buf[0] != ACK)
//		return false;
	return true;
}

/*
	<method>		version
	<description>	receive firmware version of TDU data
	<args>			None
	<return>		None
*/
void		
CTDU::version()
{
//	GOPST	gst;
    GOPST ist;

	if(!bModel){					// GOP_5084H
		txCommand("UV");			// request version sum
		eatPacket(ist);
		sVersion = ist.str;
		cout << sVersion << endl;
		strcpy(TDU_FWVer,sVersion.substr(strlen("GOP-5000HVT*A Ver ")).c_str());
		cout << TDU_FWVer << endl;
	}
	else{                           // ISD-202
        txCommand("UV");			// request version sum
		eatPacket(ist);
		sVersion = ist.str;
		cout << sVersion << endl;
		strcpy(TDU_FWVer,sVersion.substr(strlen("ISD-202VER")).c_str());
		cout << TDU_FWVer << endl;
    }
}

/*
	<method>		checksum
	<description>	receive checksum of TDU data
	<args>			None
	<return>		None
*/
int
CTDU::checksum()
{
	char	buf[32];

	if(!bModel)					// GOP_5084H
		txCommand("Sd");		// request check sum
    else
       	txCommand("Fd");
		
   if(rxAnswer(buf,30))
        cout <<  buf << endl;
    if(rxAnswer(buf,30))
	cout <<  buf << endl;
    sscanf(&buf[4],"%x",&nChecksum);
    return nChecksum;
}

/*
	<method>		sound
	<description>	play audio in designated mode	
	<args>			enum eWAV mode = W_ONCE: one time
									 W_LOOP: loop
									 W_STOP: stop 
					int no = sound #
	<return>		None
*/
void
CTDU::sound(eWAV mode, int no)
{
	ostringstream os;

	if(!bModel && bBuzzer){		// GOP_5084H
		os << "WAV " << mode << " " << no;	
		txCommand(os.str());
	}
}

/*
	<method>		backlight
	<description>	control backlight
	<args>			int ctrl = 0: dark (1)
							   1: light (31)
	<return>		None
*/
void
CTDU::backlight(int ctrl)
{
//	ostringstream os;
    char buf[80];

	if(!bModel){		// GOP_5084H
	//	sprintf(buf,"WN BRIGHTNESS ");
      //  os << "WN BRIGHTNESS ";
		if(ctrl)		// on
			sprintf(buf,"WN BRIGHTNESS 31");
       //     os << 31;
		else			// off
			sprintf(buf,"WN BRIGHTNESS 1");
       //     os << 1;
		nBL = ctrl;
		txCommand(buf);
	}
}

/*
	<method>		setEncode
	<description>	send encode
	<args>			int locale = 0: Japanese/ 1: Simplified Chinese/ 2: Traditional Chinese	
							   1: light (31)
	<return>		None
*/
void    CTDU::setEncode(int locale)
{
	switch(locale){
		default:
		case 0:
			txCommand("WN ENCODE 0");           // Japanese (English)
			break;
		case 1:
			txCommand("WN ENCODE 144");         // Simplified Chinese
			break;
		case 2:
			txCommand("WN ENCODE 152");         // Traditional Chinese
			break;
	}
}

//	Static ___________________________________________________________
/*
	<method>		getTime
	<description>	Get current time in sec.
	<args>			None
	<return>		int = current time	
*/
static int	getTime()
{
	struct	timeval		tv;
	
	gettimeofday(&tv, NULL);	// get current time
	return((int)tv.tv_sec);		// return sec.	
}
