////////////////////////////////////////////////////////////////////////////////
//	Module			: fwupIsd202.C 
//	Description		: Main module of firmware updator with ISD-202 
//
//	Compiler		: gcc version 3.2 (Red Hat Linux 8.0 3.2-7) 
//	Author			: T.Matsuzawa of JEI Matsumoto Lab.
//	E-mail			: tmatsu.jei@nifty.com
//------------------------------------------------------------------------------
//	Maintenance:
//		Mar. 1,'07	V1.0.0	Renewal for color TDU ISD-202
//		Apr.20,'07	V1.1.0	Added support of descriptor, font and error file
//		Apr.23,'07	V1.1.1	Fixed bugs releted to error messages
//		May. 5,'08	V1.1.2	Fix a bug that array size was incorrect.
//		Oct.23,'08	v1.1.3  Changed to display "warming-up" screen when there  
//							is no index file. 	
//		Dec.12, '08	v1.2.0	Renewal for EDT
//-----------------------------------------------------------------------------
////////////////////////////////////////////////////////////////////////////////
#include	<iostream>
#include 	<fstream>
#include    "../include/lisdefs.hpp"
#include 	<unistd.h>

using namespace std;

#include	"../include/pifg/libPIF.h"
#include	"../include/pifg/CLog.h"

#define	MAX_FILENAME	32
#define	C_BUTTON	0x6f
#define	C_TITLE		0x80

//	Gloval objects ___________________________________________________________

CPif*		lib;
CPrt*		pPrt;
CLog		oLog;
//int printdataOn;

//	Global methods ___________________________________________________________

//void	dispProgress(int ctrl, int n);

//	External methods _________________________________________________________

int	mcUpdate(char* index, int com);
	
//  Local methods ____________________________________________________________

static char	update(char* index);					// update MC/SC firmware
static int	updateEif(int cnd, char* fname);		// update EIF program
static int	dispInitScreen(int n, char** index);	// display initial screen
static void	dispFinalScreen(char id);				// display final screen
static int	dispStartScreen(char type,char* index);
static void	dispErrorScreen(int error);
static void dispMenu(char** index, int start, int select, int max);
static int  GetENT();
static void	resetScreen();


//	Local variables __________________________________________________________

static int			nDebug = 1;

const char *firm_name[] = {
	"MC CPU1: ",
	"MC CPU2: ",
	"SC:      ",
    "EIF:     "
};

const char*   firm_menu[]= {
    "  MC CPU1   (PWB-A) ",    // 0
    "  MC CPU2   (PWB-A) ",    // 1
    "  SC        (PWB-F) ",    // 2
    "  EIF               ",    // 3
	"  <End Flash Mode>  ",    // 4
};

static unsigned char    up[] = { 0x81, 0xa3, 0x00 };
static unsigned char    down[] = { 0x81, 0xa5, 0x00 };

static int com = 1;

static char version[] = "fwupISD202 version 2.0.0 Dec.21, 2016";

#define	indexfile	"/home/Genesis/firmware/index.upl"

static int white, black, blue, red;

//  Main module ______________________________________________________________

int		main(int argc, char** argv)
{
	char*		index[5];					// index for firmware

	char		firm_id;
	int			firm_no;
	int			i, n;
	char		indbuf1[MAX_FILENAME];
	char		indbuf2[MAX_FILENAME];
	char		indbuf3[MAX_FILENAME];
	char		indbuf4[MAX_FILENAME];
	char		indbuf5[MAX_FILENAME];
	ifstream	fin;						
	ofstream	fout;						
	bool		model = 0;

	cout << version << endl;

//  Step 0: Create a PIF library instance 

    lib = new CPif();           

    pPrt = lib->getCPrt();
	lib->p_setMode(1);		// QVGA mode
	if(argc == 2){
		com = argv[1][0] - '0';
		if((com < 1) || (com > 2))
			com = 1;
	}
	
	if(!(lib->pPanel->bModel)){
        black = GOP_BLACK;
        white = GOP_WHITE;
        blue  = GOP_BLUE;
        red   = GOP_RED;
    }
    else{
        black = ISD_BLACK;
        white = ISD_WHITE;
        blue  = BLUE;
        red   = RED;
    }

//  Step 1: Read an index file

	if(nDebug) cout << "Reading an index file: " << indexfile  << " ...\n";
	fin.open(indexfile);
//	if (fin == NULL){
    if (!fin.is_open()){
		cout << "Index file not found!\n";
    	lib->p_dispPage(1);
		exit(1);
	}
	indbuf1[0]=indbuf2[0]=indbuf3[0]=indbuf4[0]=indbuf5[0]='\0';
	index[0] = indbuf1;
	index[1] = indbuf2;
	index[2] = indbuf3;
	index[3] = indbuf4;
	index[4] = indbuf5;
	firm_no = 0;
	for(i = 0; i < 4; i++){
        if(fin.eof()) break;
        fin.getline(index[i],MAX_FILENAME-1);			// read index file
        firm_id = index[i][0];
 		if(((firm_id >= 'A') && (firm_id <= 'D')) ||
			((firm_id >= 'a') && (firm_id <= 'd')) )
			firm_no++;
	}
	fin.close();

	if(!firm_no){
		cout << "Entry not found in a Index!\n";
		exit(1);
	}

//  Step 2: Select firmware

//    index[firm_no][0] = 'H';
    strcpy(index[firm_no],firm_menu[firm_no]);

	while(1){
		n = dispInitScreen(firm_no+1,(char**)index);

//  Step 3: Update MC/SC firmware according to Index file
		if((n - 1) == 4) {
			if(dispStartScreen('H',index[n-1])) {
				system("rm -f /home/Genesis/firmware/index.upl");
				cout << "true..." << endl;
				dispFinalScreen('H');
				exit(0);
			}
			else {
				cout << "false..." << endl;
//				firm_no = 0;
				continue;
			}
		}

		firm_id = update(index[n-1]);        // Update file
		if(firm_id < 'A'){
			cout << "Failed to update!\n";
			exit(1);
		}
		else if(firm_id == 'H')
			continue;
		else if(index[n-1][0] >= 'a' && index[n-1][0] <= 'd')
			index[n-1][0] = firm_id;
//		index[firm_no][0] = 0x0;
		break;
	}


//  Step 4: Update index.file

    fout.open(indexfile,ios::out);
    for(i = 0; i < 4; i++){
        if(index[i] != 0)
            fout << index[i] << endl;
    }
    fout.close();

//  Step 5: Display Final screen

    dispFinalScreen(firm_id);

//  Step 6: Infinite Loop

    while(1){}
    return 0;
}

//  Local methods  _________________________________________________________

static char	update(char* index)
{
    int     error;
    char    firm_id;
    char    cmdbuf[16];
//    ISDST   st;
    GOPST   st;

    firm_id = *index;
//	index += 2;

//  Step 1: Check if program file exists

    ifstream    fin((index+2));                 // Open program file

    if(!fin.is_open()){
        cout << "Program file not found!\n";
        return '0';
    }
    fin.close();                            // close index file


//  Step 2: Display Initial screen

	if(!dispStartScreen(toupper(firm_id),index)){
        lib->p_clearScreen();
        return 'H';
    }

//  Step 3: Update program

    cout << "Sending " << index << endl << flush;

    if(toupper(firm_id) <= 'C'){
        if((error = mcUpdate(index,com)) != 0){
            dispErrorScreen(error);
            return '0';
        }
    }
    else{
        int cmd = toupper(firm_id) - 'D';
        updateEif(cmd,(index+2));
        lib->p_display(30,68,"Uploading ...");
        sleep(10);
   }
    cout << "\nNormal completion.\n" << flush;
    return toupper(firm_id);
}

void    dispProgress(int ctrl, int n)
{
    char progress[6];

    if(!ctrl)
        lib->p_display(100,100,"Control Program");
    else if(ctrl == 1)
        lib->p_display(100,100,"New Firmware   ");

    if(n > 100)
        n = 100;
    sprintf(progress,"%3d%%",n);
    if(lib->pPanel->bModel)
        lib->p_setColor(black,white);
    lib->p_display(190,68,progress);
}

static int	dispInitScreen(int n,char** index)
{
//    ISDST   st;
    GOPST   st;
    int     select,start;
    int     loop;
 //   int     first = true;
    char    c;

	resetScreen();

//  Select firmware in a menu

    lib->p_display(20,50,"    Select Firmware:");
    select = start = 0;
    loop = true;
	while(loop){
        dispMenu(index,start,select,n);
        lib->p_getKey(st);
	    switch(st.cmd){
		case 'U':
			if(select == 4){
				start = 0;
				select--;
			}
			else if(--select < start){
				if(select < 0){
					select = n - 1;
					start = n - 4;
				}
				else
					start--;
				if(start < 0)
					start = 0;
			}
			break;
		case 'D':
			if(++select >= start + 4)
				start++;
			if(select >= n)
				start = select = 0;
			break;
	    case 'E':
	        if(select >= 0)
	            loop = false;
	        break;
        case 'B':
            system("rm -f /home/Genesis/firmware/index.upl");
            dispFinalScreen('H');
            exit(0);
           break;
		default:
            lib->p_clearScreen();
            usleep(5000);
/*			lib->p_setColor(white,C_BUTTON);
			usleep(5000);
			lib->p_display(16,210,"      ");
			lib->p_display(84,210,"      ");
			lib->p_display(260,130,"       ");
			lib->p_display(260,90,"      ");
			lib->p_setColor(black,white);*/
			break;
	    }
	}
	return select+1;
}

static void    dispFinalScreen(char type)
{
	resetScreen();
//	lib->p_dispPage(3);
	if(lib->pPanel->bModel)
		lib->p_setColor(red,white);
	if(type == 'H'){
		lib->p_display(70,90,"End Flash Mode");
        lib->p_clearScreen();
    }
	else{
		usleep(5000000);
		lib->p_display(70,90,"Cycle Power",52);
	}
    if(toupper(type) == 'C')
        lib->p_display(30,130,"Put back to Normal mode");
    lib->p_buzzer(50);
	sleep(1);
    lib->p_buzzer(50);
}

static int	dispStartScreen(char type,char* index)
{
    int     n;
    char    version[8];
    if(lib->pPanel->bModel)
        resetScreen();
    else
        lib->p_clearScreen();
    lib->p_display(20,10,"<<< FIRMWARE UPDATE >>>");
	if(type != 'H'){
		n = strlen(index);
		version[0] = index[n-8];
		version[1] = '.';
		version[2] = index[n-7];
		version[3] = '-';
		version[4] = index[n-6];
		version[5] = index[n-5];
		version[6] = '\0';
		lib->p_display(50,88,firm_name[type - 'A']);
		lib->p_display(120,88,version);
	}
	if(lib->pPanel->bModel)
		lib->p_setColor(red,white);
	if(type == 'C')
	    {	lib->p_display(20,120,"  Check Serial Cable     ",51);
		//sleep(5);
	    }
	else if (type < 'C')
        lib->p_display(20,120," Check DIP SW & Press SW1",51);
	if(lib->pPanel->bModel)
        lib->p_setColor(black,white);
	lib->p_display(20,140,"       Enter accepts");
	lib->p_display(20,160,"        Back aborts");
	lib->p_display(20,50,"       Are you sure?");		// Are you sure ?
	if(GetENT()){
		resetScreen();
		if(type != 'H'){
			lib->p_display(100,130,firm_name[type - 'A']);
			lib->p_display(180,130,version);
			lib->p_display(30,68,"Uploading ...");	// downloading ....
		}
		return true;
	}
	return false;
}

static void    dispErrorScreen(int error)
{
    char    message[16];

    lib->p_buzzer(35,35);
	lib->p_clearScreen();
	lib->p_dispPage(3);
    sprintf(message,"Error : %02d",error);
    lib->p_display(60,60,message);
    if(lib->pPanel->bModel)
        lib->p_setColor(red,white);
    lib->p_display(60,100,"Cycle Power",52);
}

static int GetENT()
{
 //   ISDST   st;
    GOPST   st;

    while(true){
//        lib->p_getKey_FWFlash(st);
        lib->p_getKey(st);
        if(st.cmd == 'E')
            break;
        else if(st.cmd == 'B')
            return false;
    }
    return true;
}

static int updateEif(int cmd, char *fname)
{
    int     result = true;
    int     percent,oldper;
    int     current;
    int     total;
    ifstream fin;
	char	buf[64+1];

	if(cmd > 1) cmd++;
    if(!(result = lib->e_update())){
//  get file size
		fin.open(fname);
		fin.seekg(0,ios::end);
		total = (int)fin.tellg();
		fin.seekg(0,ios::beg);

//	cout << "file size : " << dec << total << " bytes.\n" << flush;
		percent = current = 0;
		oldper = -1;
		dispProgress(1,0);
		while(!fin.eof()){
			fin.getline(buf,64);	
			if(!(result = lib->e_puts(buf,0x8000))){
				cout << "TX Error\n";
				break;
			}
			current += strlen(buf);
			percent = (current * 100) / total;
			if(percent != oldper){
				dispProgress(2,percent);
				oldper = percent;
				cout << "." << flush;
			}
		}
		dispProgress(2,100);
		fin.close();
		cout << "\n" << flush;
	}
    return(result);
}       

static void dispMenu(char** index, int start, int select, int max)
{
    int i;
    int y = 80;
    if(!(lib->pPanel->bModel)){
	    lib->p_dispBlockStart();
	    lib->p_dispBlock(20,10,"<<< FIRMWARE UPDATE >>>");
	    lib->p_dispBlock(20,50,"    Select Firmware:");
    }
    if(max > 4){
        for(i = start; i < start+4; i++){
	if(lib->pPanel->bModel){
	if(i == select)
    		lib->p_setColor(white,blue);
	else
    		lib->p_setColor(black,white);
	}
	if((i == (max - 1)) && (select == (max - 1)))
        {   if(i == select){
		if(lib->pPanel->bModel)
                    lib->p_display(24,y,firm_menu[select]);
		else
                    lib->p_dispBlock(24,y,firm_menu[select],50);
		}
                else{
		if(lib->pPanel->bModel)
		    lib->p_display(24,y,firm_menu[select]);
		else
                    lib->p_dispBlock(24,y,firm_menu[select]);
		}
	}
	else {
                if(i == select){
		if(lib->pPanel->bModel)
		    lib->p_display(24,y,firm_menu[toupper(*index[i]) - 'A']);
		else
                    lib->p_dispBlock(24,y,firm_menu[toupper(*index[i]) - 'A'],50);
		}
                else{
		if(lib->pPanel->bModel)
		    lib->p_display(24,y,firm_menu[toupper(*index[i]) - 'A']);
		else
                    lib->p_dispBlock(24,y,firm_menu[toupper(*index[i]) - 'A']);
		}
	}
			if(*index[i] >= 'A' && *index[i] <= 'D'){
			if(lib->pPanel->bModel)
			   lib->p_display(24,y,'*');
			else
			   lib->p_dispBlock(24,y,'*');
			}
			y += 20;
        }
    if(lib->pPanel->bModel)
		lib->p_setColor(black,white);
    if(start){
	if(lib->pPanel->bModel)
	    lib->p_dispBlock(8,60,(char*)up);
	else
            lib->p_drawPicture(5,60,37);    //lib->p_drawPicture(5,9,37);
	}
        else
//	        lib->p_display(8,60,"  ");
        if(start < max - 4){
	if(lib->pPanel->bModel)
	    lib->p_display(8,160,(char*)down);
	else
            lib->p_drawPicture(5,150,38);
	}
    else
	        lib->p_display(8,160,"  ");
    }
    else{
        for(i = 0; i < max; i++){
	if(i == select){
		if(lib->pPanel->bModel){
    			lib->p_setColor(white,blue);
			lib->p_display(24,y,firm_menu[toupper(*index[i]) - 'A']);
		}
		else
                	lib->p_dispBlock(24,y,firm_menu[toupper(*index[i]) - 'A'],50);
	}
	else{
		if(lib->pPanel->bModel){
    			lib->p_setColor(black,white);
			lib->p_display(24,y,firm_menu[toupper(*index[i]) - 'A']);
		}else
                	lib->p_dispBlock(24,y,firm_menu[toupper(*index[i]) - 'A']);
	}
			if(*index[i] >= 'A' && *index[i] <= 'D'){
			if(lib->pPanel->bModel)
				lib->p_display(24,y,'*');
			else
				lib->p_dispBlock(24,y,'*');
			}
			y += 20;
        }
    } 
	if(!(lib->pPanel->bModel))
    	    lib->p_dispBlockStop();
}      

static void	resetScreen()
{
	lib->p_clearScreen();
    lib->p_dispPage(4);
    if(lib->pPanel->bModel)
        lib->p_setColor(black,white);
	lib->p_display(20,10,"<<< FIRMWARE UPDATE >>>");
}
