////////////////////////////////////////////////////////////////////////////////
//	Module			: McUpdate.C
//	Description		: MC/SC update module 
//
//	Compiler		: gcc version 3.2 (Red Hat Linux 8.0 3.2-7) 
//	Author			: T.Matsuzawa of JEI Matsumoto Lab.
//	E-mail			: tmatsu@po.mcci.or.jp
//------------------------------------------------------------------------------
//	Maintenance:
//		Apr. 15,'03	EV1.0	1st release
//-----------------------------------------------------------------------------
////////////////////////////////////////////////////////////////////////////////
#include	<iostream>
#include 	<fstream>
using namespace std;

#include 	<unistd.h>
#include 	<time.h>

#include	"fwupIsd202.h"
#include	"CCom.h"
#include	"CLoader.h"
#include	"fztat.h"

//	Global methods __________________________________________________________

int	mcUpdate(char* filename, int com);		// Update MC/SC firmware	

//	External methods ________________________________________________________

void	dispProgress(int flag, int percent);// Display progress (%)

//	Local methods ___________________________________________________________

static int	txControl();					// Tx control program
static int	txFirmware(char* fname);		// Tx firmware 
static int	txControlFile();				// Tx control program file
static int	txChar(char c, int verify);		// Tx a character w/ verify
static void	txLong(int l);					// TX LONG (4bytes) data

//	Local variables ___________________________________________________________
	
static	CCom*	pCOM;					// pointer to CCom object	
static	CLoader*	pCL;					// pointer to CLoader object	
	
//	Global methods __________________________________________________________

int	mcUpdate(char* filename, int com)
{
	int		ret;								// return code

//	1.	Create CCom object

	pCOM = new CCom(com);
	if(pCOM == NULL){
		cout << "Could not create CCom class object.\n";
		return ERR_CLASSNOTCREATED;
	}
	if(pCOM->getStatus() == -1){
		cout << "Could not open COM" << com << endl;
		return ERR_PORTNOTOPENED;
	}

//	2.	Tx control Program

	if((ret = txControl()) != 0){
		cout << "Error@txControl: " << hex << ret << endl;
		return ret;
	}

//	3.	Tx MC/SC firmware

	if((ret = txFirmware(filename+2)) != 0){
		cout << "Error@txFirmware: " << hex << ret << endl;
		return ret;
	}
	return ret;
}

//	Local Methods _________________________________________________________
	
// Tx control program
static int		txControl()
{
	int	ret;							// return code
	int	retry;

//	1.	bit rate adjustment

	cout << "Control program ... 1" << flush;
	for(retry = 0; retry < MAX_RETRY; retry++){
		pCOM->putchar(0x00);
		usleep(3000);
		if(pCOM->getchar(1) == 0x00)
			break;
	}
	if(retry == MAX_RETRY){
		cout << "\nCannot adjust bit rate\n";
		return ERR_NOLINK;
	}

//	2.	Establish link 
	cout << "2" << flush;
	if((ret = txChar(ST_CMD,OK)) != 0)	// send start TX code
		return ERR_NORESPONSE;				// return if error	

//	2. Tx controll program
		
	cout << "3\n" << flush;
	ret = txControlFile();	
	return ret;
}

// Tx control program file 
static int		txControlFile()
{
 	int			ret;
    int     	percent,oldper;
    int     	current;
    int     	total;
	int			i;
	char		c;

//	1. Create CLoader class
	pCL = new CLoader(MAX_CONTROL,CONTROL_OFFSET);
	if(pCL == NULL){
		cout << "Failed to create CLoader class for " << CONTROL_PROGRAM;
		cout << endl;
		return(ERR_CLASSNOTCREATED2);
	}

//	2. Load control program in S-Format 
	if((ret = pCL->loadSformat((char*)CONTROL_PROGRAM)) != 0){
		cout << "Failed to load " << CONTROL_PROGRAM;
		return ret;
	}
	total = pCL->getSize();
	cout << "  data size : " <<total << " bytes.\n";

//	3.	Tx file size

	cout << "  tx file size (upper)... \n";
	c = (total >> 8) & 0xff;
	if((ret = txChar(c,(int)c)) != 0)
		return ret;
	cout << "  tx file size (lower)... \n";
	c = total & 0xff;
	if((ret = txChar(c,(int)c)) != 0)
		return ret;

//	4.	Tx file 

	cout << "  tx binary data ... \n";
    percent = 0;
    oldper = -1;
    dispProgress(0,0);
	for(current = 0; current < total; current++){
		c = pCL->getchar();						// get a character
		if((ret = txChar(c,(int)c)) != 0)
			return ret;
		if(!(current % 10)){
			percent = (current * 100) / total;
			if(percent != oldper){
		   		dispProgress(2,percent);
		   		oldper = percent;
   	    	}
		}
	}
    dispProgress(2,100);

//	5. Wait for answer

	if(pCOM->getchar(0) == OK)
		ret = 0;
	else
		ret = ERR_ERASEFAILED;

//	6. Delete CSformat class

	delete pCL;
	return(ret);
}

// Tx firmware 
static int		txFirmware(char* fname)
{
 	int			ret;
    int     	percent,oldper;
    int     	total;
	int			i,j,k;
	int			r;
	unsigned char	c;
	unsigned char	checksum = 0;

	cout << "Firmware ... \n" << flush;
//	1. Create CLoader class
	pCL = new CLoader(MAX_APPL,0);
	if(pCL == NULL){
		cout << "Failed to create CLoader class for " << fname;
		cout << endl;
		return(ERR_CLASSNOTCREATED2);
	}

//	2. Load application (firmware) program in S-Format 
	if((ret = pCL->loadSformat(fname)) != 0){
		cout << "Failed to load " << fname;
		return ret;
	}
	total = pCL->getSize();
	r = total / TX_SIZE;					// number of block
	if(total % TX_SIZE) r++;
	total = r * TX_SIZE;					// adjust boundary
	
	cout << "  data size : " <<total << " bytes.\n";

//	3.	Establish link 
	pCOM->putchar(WT_CMD);					// send write command codes
	if((ret = txChar(NL_CMD,ACK)) != 0)
		return ERR_NORESPONSE2;

//	4.	Tx start address & data size

	cout << "  tx start address  ... \n";
	txLong(0);
	cout << "  tx data size  ... \n";
	txLong(pCL->getSize());

//	5.	Tx file 

	cout << "  tx binary data ... \n";
    percent = 0;
    oldper = -1;

    dispProgress(1,0);
	for(i = 0; i < r; i++){

		if(pCOM->getchar(0) != XON){		// handshaking
			return ERR_TIMEOUT;
		}
		pCOM->putchar(XON);

		for(j = 0; j < TX_SIZE; j++){
			c = pCL->getchar();	
			pCOM->putchar(c);	
			checksum += c;
		}
		
		if(!(i % 30)){
			percent = (i * TX_SIZE * 100) / total;
			if(percent != oldper){
			    dispProgress(2,percent);
			    oldper = percent;
   		    }
		}
	}
    dispProgress(2,100);

//	6. Send check sum
	
	pCOM->putchar(checksum ^ 0xff);

//	7. Wait for answer

	ret = pCOM->getchar(0); 

/*
	if(pCOM->getchar(0) == OK)
		ret = 0;
	else
		ret = ERR_ERASE
*/

//	8. Delete CLoader class

	delete pCL;
//	return(ret);
	return 0;
}

//	Tx character w/ verify
static int	txChar(char c, int verify){
	int	 ret = 0;
	int rx;

	verify &= 0xff;
	pCOM->putchar(c);				// send a character
	if((rx = pCOM->getchar(0)) != verify){	// verify data  	
		cout << "Verify error@txChar: " << hex << rx << " - " << verify << endl;
		ret = -1;					// verify error	
	}
	return ret;
}

//	Tx LONG (4bytes) data 
static void	txLong(int l){
	pCOM->putchar((l >> 24) & 0xff);	// send bit 31 - 24 
	pCOM->putchar((l >> 16) & 0xff);	// send bit 23 - 16 
	pCOM->putchar((l >> 8) & 0xff);		// send bit 15 -  8 
	pCOM->putchar(l & 0xff);			// send bit  7 -  0 
}
