/*	This file is part of Jenux.

    Jenux is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    Jenux is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with Jenux.  If not, see <http://www.gnu.org/licenses/>.

	Module			: uart.c 
	Description		: This is a part of device driver for PIF-II (LP-042).
	Compiler		: gcc 4.8.5++
	Email			: matsuzawa.jei@nifty.com
	Maintenance		:
		Dec. 13,'17	Released as evaluation version for CentOS7/64 
*/
#include	<linux/kernel.h>
#include	<linux/module.h>
#include	<linux/vmalloc.h>
#include	<linux/delay.h>
#include	<linux/pci.h>
#include	<linux/mm.h>
#include	<linux/sched.h>
#include	<linux/interrupt.h>
#include	<linux/io.h>
#include	<asm/uaccess.h>
#include	<linux/uaccess.h>		// added on Dec.13, 2017
#include	<asm/io.h>

#include	"common.h"
#include	"xr16L2750.h"
#include <stddef.h>

//	Global Functions __________________________________________________________

void	initUART(unsigned* uartaddr);	// Initialize UART
ssize_t uart_read(struct file *file,char *buf,size_t count,loff_t *off);
ssize_t uart_write(struct file *file,const char *buf,size_t count,loff_t *off);
void 	isrUART(int port);				// ISR for UART
int		getRxCnt(int port);				// get RX data count
int		getPAGEND(void);					// get PAGEND count
void	resetBuffer(int port);			// reset UART buffer
void	setBaud(int baud);				// set baud rate of channel-B
void	setTrigger(int trigger);		// set trigger level of channel-B
void	selectChannel(int ch);			// select channel

//	External Functions	_______________________________________________________ 

//	Local Functions ___________________________________________________________

static void		initChannel(DCB *dcb);		// Initialize a channel
static void		isrRx(DCB* dcb, int flag);	// ISR for RX
static void		isrTx(DCB* dcb);			// ISR for TX
static void		rx_do_tasklet(ulong);		// tasklet for isrRX
static void		tx_do_tasklet(ulong);		// tasklet for isrTX 

//	misc.
static void		flushBuffer(BUFFER* buf);	// flush RX/TX buffer

//	External Variables ________________________________________________________

extern int	iIntCount[];					// interrupts counters
extern int	iPAGEND;                		// PAGEND counter
extern int	iPGEDelivered;          		// PAGEND counter deliverd to user

//	Local Variables ___________________________________________________________

static unsigned*	pUARTREG;				// top of UART reg

//	Baud rate for OP (channel B) is programmable.
//	But baud rate for EIF (channel A) is fixed to 19200 bps
//
static int BaudTable[] = {
	0x30,					// 0 : 9600 bps
	0x18,					// 1 : 19200 bps
	0x0c					// 2 : 38400 bps
};

static	DCB		uartEIF;	// DCB for EIF (channel A)
static	DCB		uartOP;		// DCB for OP (channel B)

//	Declaration of tasklets (BH)

DECLARE_TASKLET(rx_tasklet1, rx_do_tasklet, (ulong)&uartEIF);
DECLARE_TASKLET(rx_tasklet2, rx_do_tasklet, (ulong)&uartOP);
DECLARE_TASKLET(tx_tasklet1, tx_do_tasklet, (ulong)&uartEIF);
DECLARE_TASKLET(tx_tasklet2, tx_do_tasklet, (ulong)&uartOP);

//	wait queue for PAGEND

static	wait_queue_head_t   pgeq;    

//	Global Functions __________________________________________________________
/*
	Name 		: initUART 
	Description	: UART Initialize routine 
	Arguments	: unsigned* uartaddr = top address of UART registers 
	Return		: None 
*/
void	initUART(unsigned* uartaddr)
{
	pUARTREG = uartaddr;			// save UART address

//	EIF	(channel A)
	uartEIF.port = 0;
	uartEIF.trigger = 2;			// 8 bytes 
	uartEIF.baud = BaudTable[1];	// 19200 bps (fixed)
	uartEIF.xon = 0;				// no software flow control
	initChannel(&uartEIF);			// initialize channel A	

//	OP	(channel B)
	uartOP.port = 1;
	uartOP.trigger = 0;				// 1 byte 
	uartOP.baud = BaudTable[2];		// 38400 bps
	uartOP.xon = 0;					// software flow control
	initChannel(&uartOP);			// initialize channel B	
}

//	System calls _____________________________________________________________
/*
	name:	uart_read
	desc:	read from device	
	args:	struct file *file = pointer to file	
			char *buf = pointer to buffer
			size_t count = size of data
			loff_t *off = file offset   
	rtrn:	int > 0	: number of RX data 
			    < 0	: error
*/
ssize_t uart_read(struct file *file,char *buf,size_t count,loff_t *off)
{
	DCB		*dcb;
	BUFFER	*rb;
    int		n;
    uchar	d;
	int		rtn;

	if(count & 0x8000) 	dcb = &uartEIF;			// This 0x1000 or lower seems to be friendlier to debugging
	else 				dcb = &uartOP;
	count &= 0xff;
	rb = &dcb->rxBuf;
//	sleep if there is no data in the buffer
	if(rb->readP == rb->writeP){
		rtn = wait_event_interruptible_timeout(dcb->rxq,(rb->readP != rb->writeP),HZ);
		if(rtn < 0)
			return -ERESTARTSYS;
		else if(!rtn)	// timeout !!
			return 0;
	}

//	OK, RX something ... then copy them to user

	if(rb->writeP > rb->readP){	
		count = min((int)count, (rb->writeP - rb->readP));
		if(copy_to_user(buf,&rb->data[rb->readP],count)){
			return -EFAULT;							// error 
		}
		rb->readP += count;
	}
	else if(rb->writeP < rb->readP){	
		count = min((int)count, (BUFSZ - rb->readP) + rb->writeP);
    	for(n = 0; n < count; n++){					// loop 	
       		d = rb->data[rb->readP++];				// get RX data 
        	if(rb->readP >= BUFSZ){					// pointer wrap?
           		rb->readP = 0;
			}
			if(put_user(d,buf+n))					// move data to user 
				return -EFAULT;						// error 
		}
    }
	else{
		count = 0;
	}
    return count;
}

/*
	name:	uart_write
	desc:	write to device	
	args:	struct file *file = pointer to file	
			char *buf = pointer to buffer
			size_t count = size of data
			loff_t *off = file offset   
	rtrn:	int > 0	: number of RX data 
			    < 0	: error
*/
ssize_t uart_write(struct file *file, const char *buf,size_t count,loff_t *off)
{
	DCB		*dcb;
	BUFFER	*tb;
	int		n;
	if(count & 0x8000) 	dcb = &uartEIF;
	else 				dcb = &uartOP;
	count &= 0x1ff;

//	if(dcb->port) printk("W%d",(int)count);  // removed in version 7.05-35x01, to reduce the entries to journal log.
	tb = &dcb->txBuf;

//	Check if the buffer empty ... If yes, then reset pointers

	if(count > BUFSZ)	n = BUFSZ;
	else				n = count;

//	printk("count = %d n = %d BUFSZ = %d\n",count, n, BUFSZ); // removed in version 7.05-35x01, to reduce the entries to journal log.

//	Calc. size of free space in the buffer

	if(tb->writeP){
		if(wait_event_interruptible(dcb->txq, (tb->writeP == 0)))
			return -ERESTARTSYS;    // error
	}

//	Copy data from user space	

	if(copy_from_user(&tb->data[tb->writeP],buf,n))
		return -EFAULT;							// error 

	tb->writeP += n;
	//writel(TX_ENABLE, IER(dcb->port));			// enable RX/TX interrupts
	writel(readl(IER(dcb->port)) | IER_TX, IER(dcb->port));			// enable RX/TX interrupts
	return(count - n);
}
//
//	Interrupt Handler ________________________________________________________
//
/*
	name:	isrUART	
	desc:	Interrupt Service Routine		
	args:	int port = port#
						0 : channel-A (EIF)
					   	1 : channel-B (O.P)	
	rtrn:	none
*/
void isrUART(int port)
{
	DCB		*dcb;
	uchar 	isr;

	if(!port)
		dcb = &uartEIF;
	else
		dcb = &uartOP;
	
	isr = READB(ISR(port)) & 0x3f;					// read ISR on channel A
//	printk("U%d<%x>",port,isr);		// removed in version 7.05-35x01, to reduce the entries to journal log.
	if(isr != ISR_NONE){
		if(isr & ISR_TXRDY) 	isrTx(dcb);		// TX intr.
		else if(isr & ISR_RXRDY)isrRx(dcb,0);	// RX intr.
		else if(isr & ISR_RXTO) isrRx(dcb,1);	// RX timeout intr.
	}
}

/*
	name:	getRxCnt
	desc:	returns nubmer od data in the RX buffer
	args:	int port = port#
	rtrn:	int = number of data
*/
int		getRxCnt(int port)				// get RX data count
{
	DCB*	dcb;

	if(!port)	dcb = &uartEIF;
	else		dcb = &uartOP;

	return abs(dcb->rxBuf.writeP - dcb->rxBuf.readP);
}

/*
	name:	getPAGEND
	desc:	returns PAGEND status counter 
	args:	none
	rtrn:	int = number of PAGEND status 
*/
int		getPAGEND() 
{
	int val;

//  sleep if there is no PAGEND

	if(iPAGEND == iPGEDelivered){
		if(wait_event_interruptible(pgeq, (iPAGEND != iPGEDelivered)))
			return -ERESTARTSYS;    // error
	}
	val = 1;
	iPGEDelivered++;
	return(val);
}

/*
	name:	resetBuffer	
	desc:	reset RX/TX buffers		
	args:	int port = port#
	rtrn:	none
*/
void	resetBuffer(int port)
{
	DCB*	dcb;

	if(!port)	dcb = &uartEIF;
	else		dcb = &uartOP;

	flushBuffer(&dcb->rxBuf);
	flushBuffer(&dcb->txBuf);
//    printk("Nwe6: FlushBuffer");		// removed in version 7.05-35x01, to reduce the entries to journal log.
}

// set baud rate of channel-B
void	setBaud(int baud)
{
	writel(RX_DISABLE,IER(1));		// Disable RX Interrupt
	uartOP.baud = BaudTable[baud];	// change baud rate
	if(baud == 1)					// TDU-910A?
		uartOP.xon = 1;				// software flow control
	else
		uartOP.xon = 0;				// software flow control
	initChannel(&uartOP);			// re-initialize channel B	
}

// set trigger level of channel-B
void	setTrigger(int trigger)
{
	writel(RX_DISABLE,IER(1));		// Disable RX Interrupt
	uartOP.trigger = trigger & 3;	// change trigger level 
	initChannel(&uartOP);			// re-initialize channel B	
}

// select channel before read & write
//	int c = 0: channel-A
//		c = 1: channel-B
void	selectChannel(int ch)
{
	//curChannel = ch;
}

//	Local Functions __________________________________________________________
/*
	Name 		: initChannel 
	Description	: Initialize channel 
	Arguments	: DCB *cdb = pointer to DCB (Device Control Block)
	Return		: None 
*/
void	initChannel(DCB *dcb)
{
	int	port = dcb->port;
	int	ctrl;

//	Set baud rate
//
	writel(LCR_DE, LCR(port));			// enable access to the divisor latch
	writel(dcb->baud, DLL(port));		// write divisor low byte 
	writel(0, DLM(port));				// write divisor high byte

//	Set Enhanced registers
//
	if(dcb->xon){
		writel(LCR_EER, LCR(port));		// Enable access to the Enhanced reg
		writel(EFR_FLOW|EFR_EN, EFR(port));	// Auto XON/XOFF flow ctrl (Tx/Rx)
		writel(XON_CHAR, XON1(port));	// XON = ^Q 
		writel(XOFF_CHAR, XOFF1(port));	// XOFF = ^S 
		writel(LCR_DEFAULT, LCR(port));	// Restore Line Control Register
    }
	else{
		writel(LCR_EER, LCR(port));		// Enable access to the Enhanced reg
		writel(EFR_EN, EFR(port));		// Enable enhanced bits 
		writel(LCR_DEFAULT, LCR(port));	// Restore Line Control Register
	}

//	Set control registers	

	writel(MCR_DEFAULT, MCR(port));		// Modem Control Register
	ctrl = FCR_FE|FCR_DME|(dcb->trigger << 6);
	writel(ctrl, FCR(port));			// FIFO Control Register
	writel(ctrl|FIFO_RESET, FCR(port));	// reset FIFO
	writel(ctrl, FCR(port));			// FIFO Control Register

//  flush RX/TX buffer

    flushBuffer(&dcb->rxBuf);
    flushBuffer(&dcb->txBuf);

//  initialize wait queue

    init_waitqueue_head(&dcb->rxq); 	// wait queue for RX
    init_waitqueue_head(&dcb->txq); 	// wait queue for TX
    if(!port)
        init_waitqueue_head(&pgeq); 	// wait queue for PAGEND

	//writel(RX_ENABLE,IER(port));		//	Enable RX Interrupt
	writel(readl(IER(dcb->port)) | IER_RX,IER(port));		//	Enable RX Interrupt
}

/*
	name:	isrRx	
	desc:	ISR for RX
	args:	DCB* dcb = pointer to DCB (Device Control Block)
			int flag = 0: RX/ 1: RX Timeout
	rtrn:	none
*/
static	void	isrRx(DCB *dcb, int flag)
{
	// Changed by angelo 2015-04-07 in reference to tmatsu change last 2009-04-16 for Fedora7 code
	//writel(readl(IER(dcb->port)) & ~RX_ENABLE, IER(dcb->port));
	writel(readl(IER(dcb->port)) & ~IER_RX, IER(dcb->port));
	if(!dcb->port){							// uart-A (EIF)
		tasklet_schedule(&rx_tasklet1);
		if(!flag)	EIF_RXRDY_CNT++;		// increase intr. counters
		else		EIF_RXTO_CNT++;
	}
	else{									// uart-B (OP)
		tasklet_schedule(&rx_tasklet2);
		if(!flag)	OP_RXRDY_CNT++;			// increase intr. counters
		else		OP_RXTO_CNT++;
	}
	// Commented by angelo 2015-04-07
	//writel(RX_DISABLE, IER(dcb->port));		// disable RX/TX interrupts
}

/*
	name:	isrTx	
	desc:	ISR for TX
	args:	DCB* dcb = pointer to DCB (Device Control Block)
	rtrn:	none
*/
static	void	isrTx(DCB* dcb)
{
	BUFFER*	buf;

	buf = &dcb->txBuf;
//	Changed by tmatsu on Jan.5, 2016
//	writel(TX_DISABLE,IER(dcb->port));			// disable TX interrupt
	writel(readl(IER(dcb->port)) & ~IER_TX,IER(dcb->port));     // disable TX interrupt
//	End of change
	if(!dcb->port){								// uart-A (EIF)
		tasklet_schedule(&tx_tasklet1);
		EIF_TXRDY_CNT++;						// increase Intr. counter
	}
	else{										// uart-B (OP)
		tasklet_schedule(&tx_tasklet2);
		OP_TXRDY_CNT++;							// increase Intr. counter
	}
}

/*
	name:	tx_do_tasklet	
	desc:	tasklet for TX interrupt	
	args:	none
	rtrn:	none
*/
static void	tx_do_tasklet(ulong arg)
{
	DCB*	dcb;
	BUFFER*	buf;

	dcb = (DCB*)arg;
	buf = &dcb->txBuf;
	
	while(buf->readP != buf->writeP){
//		printk("(%c)",buf->data[buf->readP]);	// just for debug, // removed in version 7.05-35x01, to reduce the entries to journal log.
		writel((int)buf->data[buf->readP++],THR(dcb->port));    // Tx data 
	}
	flushBuffer(buf);		// flush TX buffer
	wake_up(&dcb->txq);		// wake up user process
}

/*
	name:	rx_do_tasklet	
	desc:	tasklet for RX interrupt	
	args:	none
	rtrn:	none
*/
static void	rx_do_tasklet(ulong arg)
{
	DCB*	dcb;
	BUFFER*	buf;
	uchar	cStatus;								// status
	uchar	cData;									// RX data

	dcb = (DCB*)arg;
	buf = &dcb->rxBuf;
	while(TRUE){
		cStatus = READB(LSR(dcb->port));			// read ELSR
		if(cStatus & (LSR_FE|LSR_PE|LSR_OE)){
			cData = READB(RHR(dcb->port));			// dummy read 
			printk("COM%d Error(%x)\n",dcb->port+1,cStatus);
			break;
		}
		if(cStatus & LSR_RDR){						// rx data ? 
			cData = READB(RHR(dcb->port));			// get RX data
//			printk("[%x]",cData);	// just for debug // removed in version 7.05-35x01, to reduce the entries to journal log.
			if(!(dcb->port) && (cData == PAGEEND)){	// RX PAGEND?
				iPAGEND++;							// count up
				PAGEND_CNT++;
				wake_up_interruptible(&pgeq);		// wake up user process
			}
			else{
	        	buf->data[buf->writeP++] = cData;	// store RX data in buffer 
				if(buf->writeP >= BUFSZ)
					buf->writeP = 0;
			}
		}
		else
			break;
	}
	wake_up(&dcb->rxq);						// wake up user process
//	writel(readl(IER(dcb->port)) | IER_RX, IER(dcb->port));	// enable RX interrupt
	writel(readl(IER(dcb->port)) | RX_ENABLE, IER(dcb->port));		// enable RX interrupt
}

//	misc. __________________________________________________________________
/*
	name:	flushBuffer
	desc:	flush buffer for communication	
	args:	BUFFER* buf = pointer to BUFFER
	rtrn:	none
*/
static void	flushBuffer(BUFFER* buf)
{
	buf->writeP = buf->readP = 0;
}

